/*
 * Copyright 2021 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package com.microej.kf.util.observable;

import ej.annotation.Nullable;
import ej.kf.Feature;
import ej.kf.Feature.State;
import ej.observable.Observer;
import ej.observable.SimpleObservable;
import ej.kf.FeatureStateListener;
import ej.kf.Kernel;
import ej.kf.Module;

/**
 * Extension of {@link SimpleObservable} which can handle observers from any {@link Module}.
 */
public class KernelSimpleObservable extends SimpleObservable {

	private @Nullable FeatureStateListener featureStateListener;

	/**
	 * Creates a kernel simple observable.
	 * <p>
	 * This registers a feature state listener in the kernel. The {@link #dispose()} method has to be called once this
	 * object is not used anymore in order to unregister the feature state listener.
	 */
	public KernelSimpleObservable() {
		if (Kernel.getContextOwner() != Kernel.getInstance()) {
			throw new IllegalAccessError();
		}

		this.featureStateListener = new FeatureStateListener() {
			@Override
			public void stateChanged(@Nullable Feature feature, @Nullable State previousState) {
				if (feature != null && feature.getState() == State.STOPPED) {
					unsetObserver(feature);
				}
			}
		};
		Kernel.addFeatureStateListener(this.featureStateListener);
	}

	/**
	 * Disposes this kernel simple observable. Once it is disposed, observers can not be registered anymore.
	 */
	public void dispose() {
		FeatureStateListener featureStateListener = this.featureStateListener;
		if (featureStateListener != null) {
			unsetObserver();
			Kernel.removeFeatureStateListener(featureStateListener);
			this.featureStateListener = null;
		}
	}

	/**
	 * {@inheritDoc}
	 *
	 * @throws IllegalStateException
	 *             if this kernel simple observable has been disposed.
	 */
	@Override
	public void setObserver(Observer observer) {
		if (this.featureStateListener == null) {
			throw new IllegalStateException();
		}
		Kernel.enter();
		super.setObserver(observer);
	}

	/**
	 * Unsets the observer of this kernel simple observable if it belongs to the given module.
	 *
	 * @param module
	 *            the module which may own the observer.
	 */
	private void unsetObserver(Module module) {
		Observer observer = getObserver();
		if (observer != null && Kernel.getOwner(observer) == module) {
			unsetObserver(observer);
		}
	}
}
