/*
 * Copyright (C) 2014 The Android Open Source Project
 * Copyright (C) 2015-2019 MicroEJ Corp. This file has been modified by MicroEJ Corp.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package android.net;

import ej.basictool.ArrayTools;

/**
 * Defines a request for a network, made through {@link NetworkRequest.Builder} and or listen for changes via
 * {@link ConnectivityManager#registerNetworkCallback}.
 */
public class NetworkRequest {
	/**
	 * The {@link NetworkCapabilities} that define this request.
	 */
	public final NetworkCapabilities networkCapabilities;

	/**
	 * Identifies the request. NetworkRequests should only be constructed by the Framework and given out to applications
	 * as tokens to be used to identify the request.
	 */
	public final int requestId;

	/**
	 * Set for legacy requests and the default. Set to TYPE_NONE for none. Causes CONNECTIVITY_ACTION broadcasts to be
	 * sent.
	 */
	public final int legacyType;

	/**
	 */
	private NetworkRequest(NetworkCapabilities nc, int legacyType, int rId) {
		if (nc == null) {
			throw new NullPointerException();
		}
		this.requestId = rId;
		this.networkCapabilities = nc;
		this.legacyType = legacyType;
	}

	/**
	 * Builder used to create {@link NetworkRequest} objects. Specify the Network features needed in terms of
	 * {@link NetworkCapabilities} features
	 */
	public static class Builder {
		private final NetworkCapabilities mNetworkCapabilities = new NetworkCapabilities();

		/**
		 * Default constructor for Builder.
		 */
		public Builder() {
		}

		/**
		 * The capability to add.
		 *
		 * @param capability
		 *            The capability to add. Value is {@link NetworkCapabilities#NET_CAPABILITY_INTERNET}
		 * @return The builder to facilitate chaining builder.addCapability(...).addCapability();.
		 */
		public synchronized NetworkRequest.Builder addCapability(int capability) {
			if (!this.mNetworkCapabilities.hasCapability(capability)) {
				this.mNetworkCapabilities
				.setCapabilities(ArrayTools.add(this.mNetworkCapabilities.getCapabilities(), capability));
			}
			return this;
		}

		/**
		 * Removes (if found) the given capability from this builder instance.
		 *
		 * @param capability
		 *            The capability to remove. Value is {@link NetworkCapabilities#NET_CAPABILITY_INTERNET}
		 * @return The builder to facilitate chaining.
		 */
		public synchronized NetworkRequest.Builder removeCapability(int capability) {
			if (this.mNetworkCapabilities.hasCapability(capability)) {
				this.mNetworkCapabilities
				.setCapabilities(ArrayTools.remove(this.mNetworkCapabilities.getCapabilities(), capability));
			}
			return this;
		}

		/**
		 * Build {@link NetworkRequest} give the current set of capabilities.
		 *
		 * @return the request.
		 */
		public NetworkRequest build() {
			return new NetworkRequest(this.mNetworkCapabilities, 0, 0);
		}

	}

	@Override
	public boolean equals(Object obj) {
		if (obj instanceof NetworkRequest == false) {
			return false;
		}
		NetworkRequest that = (NetworkRequest) obj;
		return (that.legacyType == this.legacyType && that.requestId == this.requestId
				&& ((that.networkCapabilities == null && this.networkCapabilities == null)
						|| (that.networkCapabilities != null
						&& that.networkCapabilities.equals(this.networkCapabilities))));
	}

	@Override
	public int hashCode() {
		return this.requestId + (this.legacyType * 1013) + (this.networkCapabilities.hashCode() * 1051);
	}
}
