/*
 * Java
 *
 * Copyright 2022-2024 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package com.microej.microvg.test;

import static org.junit.Assert.assertEquals;

import org.junit.AfterClass;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;

import ej.bon.Constants;
import ej.microui.display.Colors;
import ej.microui.display.Display;
import ej.microui.display.GraphicsContext;
import ej.microvg.VectorFont;
import ej.microvg.VectorGraphicsPainter;

/**
 * Tests open type fonts.
 */
@SuppressWarnings("nls")
public class TestFontOpenType {

	private static final float DELTA = 0.01f;
	private static final String STRING_SQUARE = "A";
	private static final char CHAR_SQUARE = 'A';
	private static final int PADDING = 1;

	/**
	 * Starts MicroUI.
	 */
	@BeforeClass
	public static void pre() {
		TestUtilities.startMicroUI();
	}

	/**
	 * Stops MicroUI.
	 */
	@AfterClass
	public static void post() {
		TestUtilities.stopMicroUI();
	}

	/**
	 * Resets the content of the screen to black.
	 */
	@Before
	public static void preTest() {
		TestUtilities.clearScreen();
	}

	/**
	 * Tests drawing a text with a font in TTF and OTF format. (see M0092MEJAUI-2508)
	 */
	@Test
	public static void testOpenTypeWrongGlyph() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();
		VectorFont fontTTF = VectorFont.loadFont("/fonts/firstfont.ttf");
		VectorFont fontOTF = VectorFont.loadFont("/fonts/firstfont.otf");

		String string = STRING_SQUARE;
		int fontSize = 50;

		int x = display.getWidth() / 2;
		int y = display.getHeight() / 2;

		int color = Colors.WHITE;

		g.setColor(color);
		VectorGraphicsPainter.drawString(g, string, fontTTF, fontSize, x, y);
		VectorGraphicsPainter.drawString(g, string, fontOTF, fontSize, x + 100, y);
		display.flush();

		// test pixels inside the TTF character
		TestUtilities.checkArea("TTF", color, x, y, fontSize, fontSize, PADDING);
		// test pixels outside the TTF character
		TestUtilities.checkPeripheralArea("TTF", TestUtilities.BACKGROUND_COLOR, x, y, fontSize, fontSize, 20, 3);

		// test pixels inside the OTF character
		TestUtilities.checkArea("OTF", color, x + 100, y, fontSize, fontSize, PADDING);
		// test pixels outside the OTF character
		TestUtilities.checkPeripheralArea("OTF", TestUtilities.BACKGROUND_COLOR, x + 100, y, fontSize, fontSize, 20, 3);
	}

	/**
	 * Tests drawing a text with a font in TTF and OTF format. (see M0092MEJAUI-2508). This time, the drawing order is
	 * flipped, OTF first, then TTF.
	 */
	@Test
	public static void testOpenTypeWrongGlyphFlip() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();
		VectorFont fontTTF = VectorFont.loadFont("/fonts/firstfont.ttf");
		VectorFont fontOTF = VectorFont.loadFont("/fonts/firstfont.otf");

		String string = STRING_SQUARE;
		int fontSize = 50;

		int x = display.getWidth() / 2;
		int y = display.getHeight() / 2;

		int color = Colors.WHITE;

		g.setColor(color);
		VectorGraphicsPainter.drawString(g, string, fontOTF, fontSize, x + 100, y);
		VectorGraphicsPainter.drawString(g, string, fontTTF, fontSize, x, y);
		display.flush();

		// test pixels inside the TTF character
		TestUtilities.checkArea("TTF", color, x, y, fontSize, fontSize, PADDING);
		// test pixels outside the TTF character
		TestUtilities.checkPeripheralArea("TTF", TestUtilities.BACKGROUND_COLOR, x, y, fontSize, fontSize, 20, 3);

		// test pixels inside the OTF character
		TestUtilities.checkArea("OTF", color, x + 100, y, fontSize, fontSize, PADDING);
		// test pixels outside the OTF character
		TestUtilities.checkPeripheralArea("OTF", TestUtilities.BACKGROUND_COLOR, x + 100, y, fontSize, fontSize, 20, 3);
	}

	/**
	 * Tests the string width of the same font in TTF and OTF format. (see M0092MEJAUI-2435)
	 */
	@Test
	public static void testOpenTypeStringWidth() {
		VectorFont fontTTF = VectorFont.loadFont("/fonts/firstfont.ttf");
		VectorFont fontOTF = VectorFont.loadFont("/fonts/firstfont.otf");
		String string = STRING_SQUARE;
		int fontSize = 50;
		float stringWidthTTF = fontTTF.measureStringWidth(string, fontSize);
		float stringWidthOTF = fontOTF.measureStringWidth(string, fontSize);

		if (Constants.getBoolean(TestUtilities.DEBUG_CONSTANT)) {
			System.out.println("TestFontOpenType.testOpenTypeStringWidth() string width TTF: " + stringWidthTTF
					+ ", string width OTF:" + stringWidthOTF);
		}

		assertEquals(stringWidthTTF, stringWidthOTF, DELTA);
	}

	/**
	 * Tests the glyph width of the same font in TTF and OTF format. (see M0092MEJAUI-2435)
	 */
	@Test
	public static void testOpenTypeGlyphWidth() {
		VectorFont fontTTF = VectorFont.loadFont("/fonts/firstfont.ttf");
		VectorFont fontOTF = VectorFont.loadFont("/fonts/firstfont.otf");
		int fontSize = 50;
		float glyphWidthTTF = fontTTF.measureStringWidth(Character.toString(CHAR_SQUARE), fontSize);
		float glyphWidthOTF = fontOTF.measureStringWidth(Character.toString(CHAR_SQUARE), fontSize);

		if (Constants.getBoolean(TestUtilities.DEBUG_CONSTANT)) {
			System.out.println("TestFontOpenType.testOpenTypeGlyphWidth() glyph width TTF: " + glyphWidthTTF
					+ ", glyph width OTF:" + glyphWidthOTF);
		}

		assertEquals(glyphWidthTTF, glyphWidthOTF, DELTA);
	}

	/**
	 * Tests the glyph height of the same font in TTF and OTF format. (see M0092MEJAUI-2435)
	 */
	@Test
	public static void testOpenTypeGlyphHeight() {
		VectorFont fontTTF = VectorFont.loadFont("/fonts/firstfont.ttf");
		VectorFont fontOTF = VectorFont.loadFont("/fonts/firstfont.otf");
		int fontSize = 50;
		float glyphHeightTTF = fontTTF.measureStringHeight(Character.toString(CHAR_SQUARE), fontSize);
		float glyphHeightOTF = fontOTF.measureStringHeight(Character.toString(CHAR_SQUARE), fontSize);

		if (Constants.getBoolean(TestUtilities.DEBUG_CONSTANT)) {
			System.out.println("TestFontOpenType.testOpenTypeGlyphHeight() glyph height TTF: " + glyphHeightTTF
					+ ", glyph height OTF:" + glyphHeightOTF);
		}

		assertEquals(glyphHeightTTF, glyphHeightOTF, DELTA);
	}

	/**
	 * Tests the baseline position of the same font in TTF and OTF format. (see M0092MEJAUI-2435)
	 */
	@Test
	public static void testOpenTypeBaselinePosition() {
		VectorFont fontTTF = VectorFont.loadFont("/fonts/firstfont.ttf");
		VectorFont fontOTF = VectorFont.loadFont("/fonts/firstfont.otf");
		int fontSize = 50;
		float baselinePositionTTF = fontTTF.getBaselinePosition(fontSize);
		float baselinePositionOTF = fontOTF.getBaselinePosition(fontSize);

		if (Constants.getBoolean(TestUtilities.DEBUG_CONSTANT)) {
			System.out.println("TestFontOpenType.testOpenTypeBaselinePosition() baseline position TTF: "
					+ baselinePositionTTF + ", baseline position OTF:" + baselinePositionOTF);
		}

		assertEquals(baselinePositionTTF, baselinePositionOTF, DELTA);
	}
}
