/*
 * Java
 *
 * Copyright 2022-2024 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package com.microej.microvg.test;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.fail;

import org.junit.AfterClass;
import org.junit.Assert;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;

import ej.microui.display.Colors;
import ej.microui.display.Display;
import ej.microui.display.GraphicsContext;
import ej.microvg.BlendMode;
import ej.microvg.Matrix;
import ej.microvg.VectorFont;
import ej.microvg.VectorGraphicsException;
import ej.microvg.VectorGraphicsPainter;

/**
 * Test the loading and the use of external fonts.
 */
@SuppressWarnings("nls")
public class TestFontExternal {

	private static final String INTERNAL_FONT_REFERENCE = "/fonts/externalfontRef.ttf";
	private static final String EXTERNAL_FONT_ADDRESSABLE_1 = "/fonts/externalfontAddr1.ttf";
	private static final String EXTERNAL_FONT_NOT_ADDRESSABLE_1 = "/fonts/externalfontNAddr1.ttf";
	private static final String EXTERNAL_FONT_NOT_ADDRESSABLE_2 = "/fonts/externalfontNAddr2.ttf";

	private static final int PADDING = 1;
	private static final String STRING_SQUARE = "A";

	// TODO refine the test precision value when comparing floats if needed (this is an arbitrary value)
	private static final float DELTA = 0.01f;

	/**
	 * Starts MicroUI.
	 */
	@BeforeClass
	public static void pre() {
		TestUtilities.startMicroUI();
	}

	/**
	 * Stops MicroUI.
	 */
	@AfterClass
	public static void post() {
		TestUtilities.stopMicroUI();
	}

	/**
	 * Resets the content of the screen to black.
	 */
	@Before
	public static void preTest() {
		TestUtilities.clearScreen();
	}

	/**
	 * Test that loading a ttf that exists in the external memory and byte addressable.
	 */
	@Test
	public void testExistingExternalFontByteAddressable() {
		VectorFont ref = VectorFont.loadFont(INTERNAL_FONT_REFERENCE);
		VectorFont font = VectorFont.loadFont(EXTERNAL_FONT_ADDRESSABLE_1);
		Assert.assertTrue("fonts are identical", ref.getBaselinePosition(10) == font.getBaselinePosition(10));
	}

	/**
	 * Test that loading a ttf that exists in the external memory and not byte addressable.
	 */
	@Test
	public void testExistingExternalFontNotByteAddressable() {
		VectorFont ref = VectorFont.loadFont(INTERNAL_FONT_REFERENCE);
		VectorFont font = VectorFont.loadFont(EXTERNAL_FONT_NOT_ADDRESSABLE_1);
		Assert.assertTrue("fonts are identical", ref.getBaselinePosition(10) == font.getBaselinePosition(10));
	}

	/**
	 * Test that loading several external fonts and use them.
	 */
	@Test
	public void testMultipleExternalFontLoading() {
		VectorFont ref = VectorFont.loadFont(INTERNAL_FONT_REFERENCE);
		VectorFont f1 = VectorFont.loadFont(EXTERNAL_FONT_ADDRESSABLE_1);
		VectorFont f2 = VectorFont.loadFont(EXTERNAL_FONT_NOT_ADDRESSABLE_1);
		VectorFont f3 = VectorFont.loadFont(EXTERNAL_FONT_NOT_ADDRESSABLE_2);

		Assert.assertTrue("ttf ref & 1 are identical",
				ref.measureStringHeight(STRING_SQUARE, 10) == f1.measureStringHeight(STRING_SQUARE, 10));
		Assert.assertTrue("ttf ref & 2 are identical",
				ref.measureStringHeight(STRING_SQUARE, 10) == f2.measureStringHeight(STRING_SQUARE, 10));
		Assert.assertTrue("ttf ref & 3 are identical",
				ref.measureStringHeight(STRING_SQUARE, 10) == f3.measureStringHeight(STRING_SQUARE, 10));
	}

	/**
	 * Test to draw something with the external font in CPU addresses range
	 */
	@Test
	public void testDrawExternalFontAddr1() {
		// ensure to compare the same thing between the ref font and the test font
		testDrawExternalFont(VectorFont.loadFont(INTERNAL_FONT_REFERENCE));
		// testDrawExternalFont(VectorFont.loadFont(EXTERNAL_FONT_ADDRESSABLE_1));
	}

	/**
	 * Test to draw something with the external font outside CPU addresses range
	 */
	@Test
	public void testDrawExternalFontNAddr1() {
		// ensure to compare the same thing between the ref font and the test font
		testDrawExternalFont(VectorFont.loadFont(INTERNAL_FONT_REFERENCE));
		testDrawExternalFont(VectorFont.loadFont(EXTERNAL_FONT_NOT_ADDRESSABLE_1));
	}

	/**
	 * Test to draw something with the external font outside CPU addresses range
	 */
	@Test
	public void testDrawExternalFontNAddr2() {
		// ensure to compare the same thing between the ref font and the test font
		testDrawExternalFont(VectorFont.loadFont(INTERNAL_FONT_REFERENCE));
		testDrawExternalFont(VectorFont.loadFont(EXTERNAL_FONT_NOT_ADDRESSABLE_2));
	}

	/**
	 * Tests that loading a font from a path without a leading '/' fails.
	 */
	@Test
	public void testLoadFontWithoutLeadingSlashRef() {
		testLoadFontWithoutLeadingSlash(INTERNAL_FONT_REFERENCE);
	}

	/**
	 * Tests that loading a font from a path without a leading '/' fails.
	 */
	@Test
	public void testLoadFontWithoutLeadingSlashAddr1() {
		testLoadFontWithoutLeadingSlash(EXTERNAL_FONT_ADDRESSABLE_1);
	}

	/**
	 * Tests that loading a font from a path without a leading '/' fails.
	 */
	@Test
	public void testLoadFontWithoutLeadingSlashNonAddr1() {
		testLoadFontWithoutLeadingSlash(EXTERNAL_FONT_NOT_ADDRESSABLE_1);
	}

	/**
	 * Tests that loading a font from a path without a leading '/' fails.
	 */
	@Test
	public void testLoadFontWithoutLeadingSlashNonAddr2() {
		testLoadFontWithoutLeadingSlash(EXTERNAL_FONT_NOT_ADDRESSABLE_2);
	}

	private void testDrawExternalFont(VectorFont font) {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		// Draw a character with the first font
		float fontSize = 100;
		String string = STRING_SQUARE;

		int x = display.getWidth() / 2;
		int y = display.getHeight() / 2;
		Matrix matrix = new Matrix();
		matrix.setTranslate(x, y);

		int color = Colors.WHITE;

		g.setColor(color);
		VectorGraphicsPainter.drawString(g, string, font, fontSize, matrix, GraphicsContext.OPAQUE, BlendMode.SRC_OVER,
				0f);
		display.flush();

		Assert.assertEquals(fontSize * 0.2f, font.measureStringWidth(string, fontSize), DELTA);

		float targetWidth = fontSize * 0.2f;
		float targetHeight = fontSize * 0.8f;

		int startY = (int) (y + font.getBaselinePosition(fontSize) - targetHeight);

		// check inside the character
		TestUtilities.checkArea("font", color, x, startY, (int) targetWidth, (int) targetHeight, PADDING);
		// check outside the character
		TestUtilities.checkPeripheralArea("font", TestUtilities.BACKGROUND_COLOR, x, startY, (int) targetWidth,
				(int) targetHeight, 50, 3);
	}

	private void testLoadFontWithoutLeadingSlash(String validPath) {
		String invalidPath = validPath.substring(1);
		try {
			try (VectorFont font = VectorFont.loadFont(invalidPath)) {
				// Empty block, only loading the font is tested.
			}
			fail("Loading a font from a path without a leading '/' should throw a VectorGraphicsException");
		} catch (VectorGraphicsException e) {
			assertEquals("The exception should have the error code FONT_INVALID_PATH",
					VectorGraphicsException.FONT_INVALID_PATH, e.getErrorCode());
		}
	}
}
