/*
 * Copyright 2016-2020 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.mwt.animation;

import ej.basictool.ArrayTools;
import ej.basictool.ThreadUtils;
import ej.bon.Util;
import ej.microui.MicroUI;
import ej.microui.display.Display;

/**
 * An animator executes animations as fast as possible.
 * <p>
 * The animator loops on:
 * <ul>
 * <li>retrieve the current time with {@link ej.bon.Util#platformTimeMillis()},</li>
 * <li>call the {@link Animation#tick(long)} of all the registered animations,</li>
 * <li>wait for the end of the flush,</li>
 * <li>start again while at least one animation is running.</li>
 * </ul>
 * <p>
 * Only one animator can be running at any time.
 */
public class Animator {

	private final Object monitor;
	private Animation[] animations;
	private boolean hasChanged;
	private Animation[] animationsCache;

	private boolean started;
	private boolean stopRequested;

	/**
	 * Creates an animator.
	 * <p>
	 * Multiple animators may be created but only one can be running at any time.
	 */
	public Animator() {
		this.monitor = new Object();
		this.animations = new Animation[0];
		this.hasChanged = true;
		this.animationsCache = new Animation[0];
	}

	/**
	 * Starts an animation.
	 * <p>
	 * The animation tick method is then called every period.
	 *
	 * @param animation
	 *            the animation to start.
	 * @see Animation#tick(long)
	 */
	public void startAnimation(Animation animation) {
		synchronized (this.monitor) {
			this.animations = ArrayTools.add(this.animations, animation);
			this.hasChanged = true;
			start();
		}
	}

	/**
	 * Stops an animation.
	 * <p>
	 * The animation tick method will not be called anymore by this animator.
	 *
	 * @param animation
	 *            the animation to stop.
	 */
	public void stopAnimation(Animation animation) {
		synchronized (this.monitor) {
			this.animations = ArrayTools.remove(this.animations, animation);
			this.hasChanged = true;
			if (this.animations.length == 0) {
				this.stopRequested = true;
				this.animationsCache = new Animation[0];
			}
		}
	}

	/**
	 * Gets a cache of the set of animations to avoid concurrent modification.
	 *
	 * @return the cached animations set.
	 */
	private Animation[] getAnimations() {
		synchronized (this.monitor) {
			if (this.hasChanged) {
				this.hasChanged = false;
				this.animationsCache = this.animations.clone();
			}
		}
		return this.animationsCache; // NOSONAR the array is only used inside its instance.
	}

	private void tick() {
		long currentTimeMillis = Util.platformTimeMillis();
		Animation[] animations = getAnimations();
		for (Animation animation : animations) {
			assert animation != null;
			try {
				if (!animation.tick(currentTimeMillis)) {
					stopAnimation(animation);
				}
			} catch (Exception e) {
				ThreadUtils.handleUncaughtException(e);
				stopAnimation(animation);
			}
		}
	}

	// MUST be called while owning the monitor on this.animations
	private void start() {
		this.stopRequested = false;
		if (!this.started) {
			this.started = true;
			Runnable run = new Runnable() {
				@Override
				public void run() {
					runTick(this);
				}

			};
			MicroUI.callSerially(run);
		}
	}

	private void runTick(Runnable runnable) {
		tick();
		boolean restart;
		synchronized (this.monitor) {
			if (this.stopRequested) {
				this.started = false;
				restart = false;
			} else {
				restart = true;
			}
		}
		if (restart) {
			Display display = Display.getDisplay();
			// Make sure that the callOnFlushCompleted will be executed.
			display.requestFlush();
			display.callOnFlushCompleted(runnable);
		}
	}

}
