/*
 * Java
 *
 * Copyright 2018-2024 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.bluetooth.util.services.cts;

import java.util.Calendar;

import ej.bluetooth.BluetoothAttribute;
import ej.bluetooth.BluetoothCharacteristic;
import ej.bluetooth.BluetoothConnection;
import ej.bluetooth.BluetoothService;
import ej.bluetooth.BluetoothStatus;
import ej.bluetooth.listeners.impl.DefaultRemoteServiceListener;
import ej.bluetooth.util.AttributeNotFoundException;
import ej.bluetooth.util.ServiceHelper;
import ej.bon.ByteArray;

/**
 * The <code>CurrentTimeClient</code> class represents a current time client.
 */
public abstract class CurrentTimeClient extends DefaultRemoteServiceListener {

	private final BluetoothConnection connection;

	private final BluetoothService service;
	private final BluetoothCharacteristic currentTimeChar;
	private final BluetoothCharacteristic localTimeInfoChar;

	/**
	 * Creates a current time client, using the current time service provided by a remote device and the connection to
	 * this device.
	 *
	 * @param connection
	 *            the connection to the device.
	 * @param service
	 *            the current time service provided by the device.
	 * @throws AttributeNotFoundException
	 *             if one of the mandatory attributes of the service is missing.
	 */
	public CurrentTimeClient(BluetoothConnection connection, BluetoothService service)
			throws AttributeNotFoundException {
		this.connection = connection;

		this.service = service;
		this.currentTimeChar = ServiceHelper.getCharacteristic(service, CurrentTimeConstants.CURRENT_TIME_UUID);
		this.localTimeInfoChar = ServiceHelper.getCharacteristic(service, CurrentTimeConstants.LOCAL_TIME_INFO_UUID);
	}

	/**
	 * Starts this current time client.
	 */
	public void start() {
		this.service.setRemoteListener(this);
	}

	/**
	 * Stops this current time client.
	 */
	public void stop() {
		this.service.setRemoteListener(new DefaultRemoteServiceListener());
	}

	/**
	 * Requests the current time and local time to the server. The {@link #onCurrentTimeUpdate(long)} and
	 * {@link #onLocalTimeUpdate(long)} methods are called once the time information is received from the server.
	 */
	public void requestTime() {
		this.connection.sendReadRequest(this.currentTimeChar);
		this.connection.sendReadRequest(this.localTimeInfoChar);
	}

	@Override
	public void onReadCompleted(BluetoothConnection connection, BluetoothAttribute attribute, byte status,
			byte[] value) {
		if (status == BluetoothStatus.OK) {
			if (attribute == this.currentTimeChar) {
				onCurrentTimeUpdate(readCurrentTime(value));
			} else if (attribute == this.localTimeInfoChar) {
				onLocalTimeUpdate(readLocalTimeOffset(value));
			}
		}
	}

	/**
	 * Called when the current time is received.
	 *
	 * @param currentTime
	 *            the current time (in milliseconds).
	 */
	protected abstract void onCurrentTimeUpdate(long currentTime);

	/**
	 * Called when the local time is received.
	 *
	 * @param localTimeOffset
	 *            the local time offset (in milliseconds).
	 */
	protected abstract void onLocalTimeUpdate(long localTimeOffset);

	private static long readCurrentTime(byte[] value) {
		int year = (ByteArray.readShort(value, 0, ByteArray.LITTLE_ENDIAN) & 0xFFFF);
		int month = (value[2] & 0xFF);
		int day = (value[3] & 0xFF);
		int hour = (value[4] & 0xFF);
		int minute = (value[5] & 0xFF);
		int second = (value[6] & 0xFF);
		int dayOfWeek = (value[7] & 0xFF);
		int fractionsOfSecond = (value[8] & 0xFF);

		Calendar calendar = Calendar.getInstance();

		calendar.set(Calendar.YEAR, year);
		calendar.set(Calendar.MONTH, month - 1);
		calendar.set(Calendar.DAY_OF_MONTH, day);
		calendar.set(Calendar.HOUR_OF_DAY, hour);
		calendar.set(Calendar.MINUTE, minute);
		calendar.set(Calendar.SECOND, second);

		if (dayOfWeek == 7) {
			calendar.set(Calendar.DAY_OF_WEEK, Calendar.SUNDAY);
		} else {
			calendar.set(Calendar.DAY_OF_WEEK, dayOfWeek + 1);
		}

		calendar.set(Calendar.MILLISECOND, fractionsOfSecond * 1000 / 256);

		return calendar.getTimeInMillis();
	}

	private static long readLocalTimeOffset(byte[] value) {
		int timezoneOffset = (value[0] & 0xFF);
		int dstOffset = (value[1] & 0xFF);
		return (long) (timezoneOffset + dstOffset) * 15 * 60 * 1000;
	}
}
