/*
 * Java
 *
 * Copyright 2021-2022 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package ej.hoka.http;

import java.util.Date;

import ej.annotation.Nullable;

/**
 * Web Cookie.
 */
public class Cookie {

	/**
	 * Same site values.
	 */
	public enum SameSite {
		/**
		 * The browser sends the cookie only for same-site requests (that is, requests originating from the same site
		 * that set the cookie). If the request originated from a different URL than the current one, no cookies with
		 * the SameSite=Strict attribute are sent.
		 */
		Strict, // NOSONAR
		/**
		 * The cookie is not sent on cross-site requests, such as calls to load images or frames, but is sent when a
		 * user is navigating to the origin site from an external site (e.g. if following a link). This is the default
		 * behavior if the SameSite attribute is not specified.
		 */
		Lax, // NOSONAR
		/**
		 * The browser sends the cookie with both cross-site and same-site requests. The Secure attribute must also be
		 * set when SameSite=None!
		 */
		None; // NOSONAR
	}

	@Nullable
	private final String name;
	@Nullable
	private final String value;
	@Nullable
	private String domain;
	@Nullable
	private String path;
	private int maxAge;
	@Nullable
	private Date expires;
	private boolean secured;
	private boolean httpOnly;
	@Nullable
	private SameSite sameSite;

	Cookie(@Nullable String name, @Nullable String value, @Nullable String domain, @Nullable String path, int maxAge, // NOSONAR
			@Nullable Date expires, boolean secured, boolean httpOnly, @Nullable SameSite sameSite) {
		this(name, value, maxAge);
		this.domain = domain;
		this.path = path;
		this.maxAge = maxAge;
		this.expires = expires != null ? (Date) expires.clone() : null;
		this.secured = secured;
		this.httpOnly = httpOnly;
		this.sameSite = sameSite;
	}

	Cookie(@Nullable String name, @Nullable String value, int maxAge) {
		this.name = name;
		this.value = value;
		this.maxAge = maxAge;
	}

	/**
	 * Configures and builds a cookie.
	 *
	 * @return builder instance to continue configuring the cookie or building the instance by calling
	 *         {@link Builder#build()}
	 */
	public static Builder builder() {
		return new Builder();
	}

	/**
	 * Cookie builder.
	 */
	public static final class Builder {
		@Nullable
		private String name = null;
		@Nullable
		private String value = null;
		@Nullable
		private Date expires = null;
		@Nullable
		private String domain = null;
		@Nullable
		private String path = null;
		private int maxAge = 0;
		private boolean secured = false;
		private boolean httpOnly = false;
		@Nullable
		private SameSite sameSite = null;

		private Builder() {
			// no-op
		}

		/**
		 * A {@code <cookie-name> } can be any US-ASCII characters, except control characters, spaces, or tabs. It also
		 * must not contain a separator character like the following: {@code ( ) < > @ , ; : \ " / [ ] ? = { } }.
		 *
		 * @param name
		 *            cookie name
		 * @return builder instance to continue configuring the cookie or building the instance by calling
		 *         {@link Builder#build()}
		 */
		public Builder name(@Nullable String name) {
			if (name == null || name.isEmpty()) {
				throw new IllegalArgumentException();
			}
			this.name = name;
			return this;
		}

		/**
		 * A {@code <cookie-value> } can optionally be wrapped in double quotes and include any US-ASCII characters
		 * excluding control characters, Whitespace, double quotes, comma, semicolon, and backslash. Encoding: Many
		 * implementations perform URL encoding on cookie values, however it is not required per the RFC specification.
		 * It does help satisfying the requirements about which characters are allowed for {@code <cookie-value> }
		 * though.
		 *
		 * @param value
		 *            cookie value
		 * @return builder instance to continue configuring the cookie or building the instance by calling
		 *         {@link Builder#build()}
		 */
		public Builder value(@Nullable String value) {
			if (value == null || value.isEmpty()) {
				throw new IllegalArgumentException();
			}
			this.value = value;
			return this;
		}

		/**
		 *
		 * {@code Expires=<date>} Optional
		 *
		 * The maximum lifetime of the cookie as an HTTP-date timestamp.
		 *
		 * If unspecified, the cookie becomes a session cookie. A session finishes when the client shuts down, and
		 * session cookies will be removed.
		 *
		 * Warning:
		 *
		 * Many web browsers have a session restore feature that will save all tabs and restore them next time the
		 * browser is used. Session cookies will also be restored, as if the browser was never closed.
		 *
		 * When an Expires date is set, the deadline is relative to the client the cookie is being set on, not the
		 * server.
		 *
		 * @param expires
		 *            the maximum lifetime of the cookie as an HTTP-date timestamp
		 * @return builder instance to continue configuring the cookie or building the instance by calling
		 *         {@link Builder#build()}
		 */
		public Builder expires(@Nullable Date expires) {
			if (expires == null) {
				throw new IllegalArgumentException();
			}
			this.expires = (Date) expires.clone();
			return this;
		}

		/**
		 * {@code Domain=<domain-value>} Optional
		 *
		 * Host to which the cookie will be sent. If omitted, defaults to the host of the current document URL, not
		 * including subdomains. Contrary to earlier specifications, leading dots in domain names (.example.com) are
		 * ignored. Multiple host/domain values are not allowed, but if a domain is specified, then subdomains are
		 * always included.
		 *
		 * @param domain
		 *            Host to which the cookie will be sent.
		 * @return builder instance to continue configuring the cookie or building the instance by calling
		 *         {@link Builder#build()}
		 */
		public Builder domain(@Nullable String domain) {
			if (domain == null || domain.isEmpty()) {
				throw new IllegalArgumentException();
			}
			this.domain = domain;
			return this;
		}

		/**
		 * {@code Path=<path-value>} Optional
		 *
		 * A path that must exist in the requested URL, or the browser won't send the Cookie header. The forward slash
		 * (/) character is interpreted as a directory separator, and subdirectories will be matched as well: for
		 * Path=/docs, /docs, /docs/Web/, and /docs/Web/HTTP will all match.
		 *
		 * @param path
		 *            A path that must exist in the requested URL, or the browser won't send the Cookie header
		 * @return builder instance to continue configuring the cookie or building the instance by calling
		 *         {@link Builder#build()}
		 */
		public Builder path(@Nullable String path) {
			if (path == null || path.isEmpty()) {
				throw new IllegalArgumentException();
			}
			this.path = path;
			return this;
		}

		/**
		 * {@code Max-Age=<number>} Optional
		 *
		 * Number of seconds until the cookie expires. A zero or negative number will expire the cookie immediately. If
		 * both Expires and Max-Age are set, Max-Age has precedence.
		 *
		 * @param maxAge
		 *            seconds until the cookie expires
		 * @return builder instance to continue configuring the cookie or building the instance by calling
		 *         {@link Builder#build()}
		 */
		public Builder maxAge(int maxAge) {
			this.maxAge = maxAge;
			return this;
		}

		/**
		 * Secure Optional
		 *
		 * Cookie is only sent to the server when a request is made with the https: scheme (except on localhost), and
		 * therefore is more resistent to man-in-the-middle attacks. Note: Do not assume that Secure prevents all access
		 * to sensitive information in cookies (session keys, login details, etc.). Cookies with this attribute can
		 * still be read/modified with access to the client's hard disk, or from JavaScript if the HttpOnly cookie
		 * attribute is not set.
		 *
		 * Note: Insecure sites (http:) can't set cookies with the Secure attribute (since Chrome 52 and Firefox 52).
		 * For Firefox, the https: requirements are ignored when the Secure attribute is set by localhost (since Firefox
		 * 75).
		 *
		 * @return builder instance to continue configuring the cookie or building the instance by calling
		 *         {@link Builder#build()}
		 */
		public Builder secure() {
			this.secured = true;
			return this;
		}

		/**
		 *
		 * HttpOnly Optional
		 *
		 * Forbids JavaScript from accessing the cookie, for example, through the Document.cookie property. Note that a
		 * cookie that has been created with HttpOnly will still be sent with JavaScript-initiated requests, e.g. when
		 * calling XMLHttpRequest.send() or fetch(). This mitigates attacks against cross-site scripting (XSS).
		 *
		 * @return builder instance to continue configuring the cookie or building the instance by calling
		 *         {@link Builder#build()}
		 */
		public Builder httpOnly() {
			this.httpOnly = true;
			return this;
		}

		/**
		 * {@code SameSite=<samesite-value>} Optional
		 *
		 * Controls whether a cookie is sent with cross-origin requests, providing some protection against cross-site
		 * request forgery attacks (CSRF).
		 *
		 * Inline options are:
		 *
		 * Strict:
		 *
		 * The browser sends the cookie only for same-site requests (that is, requests originating from the same site
		 * that set the cookie). If the request originated from a different URL than the current one, no cookies with
		 * the SameSite=Strict attribute are sent.
		 *
		 * Lax:
		 *
		 * The cookie is not sent on cross-site requests, such as calls to load images or frames, but is sent when a
		 * user is navigating to the origin site from an external site (e.g. if following a link). This is the default
		 * behavior if the SameSite attribute is not specified.
		 *
		 * None:
		 *
		 * The browser sends the cookie with both cross-site and same-site requests. The Secure attribute must also be
		 * set when SameSite=None!
		 *
		 * @param sameSite
		 *            Controls whether a cookie is sent with cross-origin requests, providing some protection against
		 *            cross-site request forgery attacks (CSRF)
		 * @return builder instance to continue configuring the cookie or building the instance by calling
		 *         {@link Builder#build()}
		 */
		public Builder sameSite(@Nullable SameSite sameSite) {
			if (sameSite == null) {
				throw new IllegalArgumentException();
			}
			this.sameSite = sameSite;
			return this;
		}

		/**
		 * Builds the cookie instance.
		 *
		 * @return {@link Cookie} instance
		 */
		public Cookie build() {
			return new Cookie(this.name, this.value, this.domain, this.path, this.maxAge, this.expires, this.secured,
					this.httpOnly, this.sameSite);
		}

	}

	/**
	 * Gets the name.
	 *
	 * @return the name.
	 */
	@Nullable
	public String getName() {
		return this.name;
	}

	/**
	 * Gets the value.
	 *
	 * @return the value.
	 */
	@Nullable
	public String getValue() {
		return this.value;
	}

	/**
	 * Gets the expiration date.
	 *
	 * @return the expiration date (may be null).
	 */
	@Nullable
	public Date getExpires() {
		return this.expires != null ? (Date) this.expires.clone() : null;
	}

	/**
	 * Gets the domain.
	 *
	 * @return the domain (may be null).
	 */
	@Nullable
	public String getDomain() {
		return this.domain;
	}

	/**
	 * Gets the path.
	 *
	 * @return the path (may be null).
	 */
	@Nullable
	public String getPath() {
		return this.path;
	}

	/**
	 * Gets the max age of the cookie.
	 *
	 * @return the maxAge.
	 */
	public int getMaxAge() {
		return this.maxAge;
	}

	/**
	 * Checks if the cookie is secured.
	 *
	 * @return true if the cookie is secured, false otherwise.
	 */
	public boolean isSecure() {
		return this.secured;
	}

	/**
	 * Checks if the cookie is for http only.
	 *
	 * @return true if the cookie is for http only, false otherwise.
	 */
	public boolean isHttpOnly() {
		return this.httpOnly;
	}

	/**
	 * Gets the same site policy.
	 *
	 * @return the same site policy value (may be null).
	 */
	@Nullable
	public SameSite getSameSite() {
		return this.sameSite;
	}

	@Override
	public String toString() {
		final StringBuilder cookie = new StringBuilder();
		cookie.append(this.name).append("=").append(this.value); //$NON-NLS-1$

		Date expires = this.expires;
		if (expires != null) {
			cookie.append("; Expires=").append(expires.toString()); //$NON-NLS-1$
		}

		if (this.maxAge != 0) {
			cookie.append("; Max-Age=").append(this.maxAge); //$NON-NLS-1$
		}

		if (this.domain != null && !this.domain.isEmpty()) {
			cookie.append("; Domain=").append(this.domain); //$NON-NLS-1$
		}

		if (this.path != null && !this.path.isEmpty()) {
			cookie.append("; Path=").append(this.path); //$NON-NLS-1$
		}

		if (this.secured) {
			cookie.append("; Secure"); //$NON-NLS-1$
		}

		if (this.httpOnly) {
			cookie.append("; HttpOnly"); //$NON-NLS-1$
		}

		SameSite site = this.sameSite;
		if (site != null) {
			cookie.append("; SameSite=").append(site.name()); //$NON-NLS-1$
		}

		return cookie.toString();
	}

}
