/*
 * Java
 *
 * Copyright 2018-2020 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.bluetooth;

/**
 * The <code>BluetoothPermissions</code> class enumerates the values for attribute permissions.
 * <p>
 * A read flag and a write flag may be combined to create a set of permissions.
 * <p>
 * Attribute permissions are checked when a client wants to access an attribute. The permissions are only used by the
 * server and they are not sent to the client.
 * <p>
 * Refer to «Core Specification Vol 3, Part F, 3.2.5 Attribute Permissions».
 */
public class BluetoothPermissions {

	/** No access. */
	public static final byte NONE = 0x00;

	/** Read access, no encryption required. */
	public static final byte READ = 0x01;
	/** Read access, encryption required. */
	public static final byte READ_ENCRYPT = 0x02;
	/** Read access, authentication required. */
	public static final byte READ_AUTH = 0x04;

	/** Write access, no encryption required. */
	public static final byte WRITE = 0x08;
	/** Write access, encryption required. */
	public static final byte WRITE_ENCRYPT = 0x10;
	/** Write access, authentication required. */
	public static final byte WRITE_AUTH = 0x20;

	/** Read and write access, no encryption required. */
	public static final byte RW = READ | WRITE;
	/** Read and write access, encryption required. */
	public static final byte RW_ENCRYPT = READ_ENCRYPT | WRITE_ENCRYPT;
	/** Read and write access, authentication required. */
	public static final byte RW_AUTH = READ_AUTH | WRITE_AUTH;

	/**
	 * Private constructor.
	 */
	private BluetoothPermissions() {
		throw new RuntimeException();
	}
}
