/*
 * Java
 *
 * Copyright 2019-2021 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.hoka.session;

import java.util.Base64;
import java.util.Date;
import java.util.Iterator;
import java.util.Random;

import ej.annotation.Nullable;
import ej.basictool.map.PackedMap;
import ej.bon.Timer;
import ej.bon.TimerTask;

/**
 * Handle sessions and stores active sessions in a database.
 */
public class SessionHandler {

	private static final long DEFAULT_SESSION_LIFETIME = 60L * 60; // 1 hour

	/**
	 * Number of milliseconds in a second, for s to ms conversion.
	 */
	private static final int MILLISECONDS_IN_SECOND = 1000;

	private static final int TOKEN_SIZE = 128;

	private final Random randomNumberGenerator;
	private final long sessionLifetime; // in seconds

	private final PackedMap<String, Session> sessions;

	/**
	 * Timer for zombie sessions cleaning task.
	 */
	private final Timer timer = new Timer();

	/**
	 * Constructs a {@link SessionHandler} with 1-hour-long sessions.
	 *
	 * Use a secure {@link Random} implementation (see java.security.SecureRandom).
	 *
	 * @param random
	 *            the random number generator used to create session IDs.
	 *
	 *
	 */
	public SessionHandler(Random random) {
		this(random, DEFAULT_SESSION_LIFETIME);
	}

	/**
	 * Constructs a {@link SessionHandler} provided sessionLifetime sessions and using an in-memory database.
	 *
	 * Use a secure {@link Random} implementation (see java.security.SecureRandom).
	 *
	 * @param random
	 *            the random number generator used to create session IDs.
	 * @param sessionLifetime
	 *            the time before a session is considered invalid in seconds.
	 *
	 *
	 */
	public SessionHandler(Random random, long sessionLifetime) {
		this.randomNumberGenerator = random;
		this.sessionLifetime = sessionLifetime;
		this.sessions = new PackedMap<>();

		/**
		 * Clean zombie sessions data base every 15 minutes
		 */
		final TimerTask cleanTask = new TimerTask() {

			@Override
			public void run() {
				synchronized (SessionHandler.this.sessions) {
					Iterator<Session> sessionIterator = SessionHandler.this.sessions.values().iterator();
					while (sessionIterator.hasNext()) {
						Session s = sessionIterator.next();
						if (s.hasExpired()) {
							sessionIterator.remove();
						}
					}
				}

			}
		};

		this.timer.scheduleAtFixedRate(cleanTask, new Date(), sessionLifetime * MILLISECONDS_IN_SECOND);
	}

	/**
	 * Creates a new session.
	 *
	 * @return the generated session ID.
	 */
	public Session newSession() {
		Session newSession = new Session(generateSessionID(), generateExpiration());
		this.sessions.put(newSession.getId(), newSession);
		return newSession;
	}

	/**
	 * Gets the session associated to the given ID, if any.
	 *
	 * @param sessionID
	 *            session id
	 * @return session with the provided ID, or <code>null</code> if no session has been found for this ID or if the
	 *         found session has expired.
	 */
	@Nullable
	public Session getSession(String sessionID) {
		synchronized (this.sessions) {
			final Session session = this.sessions.get(sessionID);
			if (session == null) {
				return null;
			}

			if (session.hasExpired()) {
				this.sessions.remove(session.getId());
				return null;
			}

			return session;
		}
	}

	/**
	 * Refreshes the expiration date of the session identified by <code>sessionID</code>.
	 *
	 * @param sessionID
	 *            the identifier of the session.
	 */
	public void refresh(String sessionID) {
		synchronized (this.sessions) {
			if (this.sessions.get(sessionID) != null) {
				Session session = this.sessions.get(sessionID);
				if (session != null) {
					session.setExpiration(generateExpiration());
				}
			}
		}
	}

	/**
	 * Removes the session identified by <code>sessionID</code> from the active sessions.
	 *
	 * @param sessionID
	 *            the identifier of the session.
	 * @return {@code false} if no sessions are referenced by <code>sessionID</code>, {@code true} otherwise
	 */
	public boolean removeSession(String sessionID) {
		synchronized (this.sessions) {
			final Session session = this.sessions.get(sessionID);
			if (session != null) {
				this.sessions.remove(sessionID);
				return true;
			}

			return false;
		}
	}

	/**
	 * Generates a new session ID encoded in base64.
	 *
	 * @return the generated session ID.
	 */
	protected String generateSessionID() {
		byte[] sidBytes = new byte[TOKEN_SIZE];
		this.randomNumberGenerator.nextBytes(sidBytes);
		return Base64.getEncoder().encodeToString(sidBytes);
	}

	/**
	 * Generates the expiration date using the current real time.
	 *
	 * @return the generated expiration date.
	 *
	 * @see System#currentTimeMillis()
	 */
	protected long generateExpiration() {
		return System.currentTimeMillis() + this.sessionLifetime;
	}

}
