/*
 * Java
 *
 * Copyright 2010-2017 IS2T. All rights reserved.
 * IS2T PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package ej.microui.display;

/**
 * The <code>GraphicsContext</code> class offers basic drawing facilities, to render lines, rectangles, polygons, arcs
 * and text.<br>
 * <br>
 * <code>GraphicsContext</code> uses 24-bit RGB color. Each color: red, green and blue is defined with an 8-bit value.
 * <br>
 * Not all displays may support such color depth. Therefore the implementation is in charge of mapping application
 * colors to the most appropriate available colors.<br>
 * <br>
 * A <code>GraphicsContext</code> object may be used either to
 * <ul>
 * <li>paint on a display in the "normal" rendering procedure (using <code>paint(GraphicsContext)</code> methods), or
 * </li>
 * <li>draw on a mutable image, or</li>
 * <li>directly draw on a display bypassing the "normal" rendering mechanism.</li>
 * </ul>
 * When a visible object has to be painted on a <code>Display</code>, the <code>paint</code> method is given a
 * <code>GraphicsContext</code> as argument and should use it to render the visible object.<br>
 * <br>
 * A <code>GraphicsContext</code> may be requested for a mutable image. This graphics context can be used to draw in the
 * image.<br>
 * <br>
 * Direct drawing on a display can be done from application by retrieving a <code>GraphicsContext</code> with
 * {@link Display#getNewGraphicsContext()} or {@link Display#getNewExplicitFlush()}. Using this mechanism does not
 * ensure drawings will be performed before, during or after the current <code>paint()</code>, since it bypasses the
 * serialization system events.<br>
 * <br>
 * Drawing text relies on available fonts. Text can be drawn using {@link #drawChar(char, int, int, int) drawChar},
 * {@link #drawChars(char[], int, int, int, int, int) drawChars}, {@link #drawString(String, int, int, int) drawString}
 * or {@link #drawSubstring(String, int, int, int, int, int) drawSubstring}. Characters are drawn with the current color
 * of the <code>GraphicsContext</code> object.<br>
 * <br>
 * The coordinate system is as follows:
 * <ul>
 * <li>origin is at the upper left corner of the destination.</li>
 * <li>X-axis is positive towards the right.</li>
 * <li>Y-axis is positive downwards.</li>
 * </ul>
 * A coordinate does not map a pixel, but rather the location between pixels. For instance, the first pixel in the upper
 * left corner matches a square of coordinates: <code>(0,0), (1,0), (1,1) and (0,1)</code>. The call (where g is a
 * <code>GraphicsContext</code>) <code>g.fillRect(1,0,2,3)</code> paints six pixels.<br>
 * <br>
 * Two different stroke styles may be used when drawing lines, arcs or rectangles: either <code>SOLID</code> or
 * <code>DOTTED</code>. Stroke style has no effect on fill, text and image handling.<br>
 * <br>
 * The <code>SOLID</code> stroke style allows drawing with a one-pixel wide pen. Drawing at a specific coordinate fills
 * the adjacent down-right pixel. For instance, although the next line has a width of 1,
 * <code>g.drawLine(0,0,1,0)</code> draws 2 pixels: the upper-left corner of the display and its adjacent right pixel.
 * <br>
 * <br>
 * The <code>DOTTED</code> stroke style allows drawing a subset of the pixels that would have been drawn with the
 * <code>SOLID</code> stroke style. Length and frequency of dots is implementation dependent and, as a result, so are
 * the drawn pixels. Note that end of lines or end of arcs, as well as the corner of rectangles may not be drawn with
 * the <code>DOTTED</code> stroke style.<br>
 * <br>
 * One important remark has to be made about rectangle drawing and filling. Drawing a rectangle with the code:<br>
 * <code>drawRect(x,y,w,h);</code><br>
 * is equivalent to the following code sequence:<br>
 * <code>drawLine(x,y,x+w,y);</code><br>
 * <code>drawLine(x+w,y,x+w,y+h);</code><br>
 * <code>drawLine(x+w,y+h,x,y+h);</code><br>
 * <code>drawLine(x,y+h,x,y);</code><br>
 * <br>
 * In addition, the following code:<br>
 * <code>fillRect(x,y,w,h);</code><br>
 * results in filling rectangle area which differs from the rectangle drawn by <code>drawRect(x,y,w,h)</code>. Indeed,
 * the filled area counts <code>w*h</code> pixels, whereas the area delimited by <code>drawRect(x,y,w,h)</code> counts
 * <code>(w+1)*(h+1)</code> pixels.<br>
 * A filled area must overlap exactly or be contiguous to its matching drawn area. That is to say that there must be no
 * blank space between a filled area and its matching drawn area and that the filled area must not be out of the bounds
 * of the drawn area.<br>
 * <br>
 * Note that the exact number of pixels drawn by <code>drawLine()</code> and <code>drawArc()</code> are implementation
 * dependent.<br>
 * <br>
 * A <code>GraphicsContext</code> defines a clipping zone which specifies the destination area that can be modified by
 * calls to the <code>GraphicsContext</code>. The clipping zone can be set by the application but is more commonly set
 * by the UI framework. The clipping zone may be empty (i.e. its size is zero), in that case, every rendering operation
 * will have no effect. It may also be out of the bounds of the destination, in which case every rendering operation out
 * of the range of the destination is ignored. Modification of the coordinate system (with the method
 * <code>translate</code> for instance) has no effect on the clipping zone.<br>
 * <br>
 * When positioning a visible object (text or image for instance) into a drawable area, a coordinate <code>(x,y)</code>
 * location or anchor point is used. In addition it is possible to express how the object is set around the anchor
 * point. Several constants have been thus defined; they can be combined bit-wise to precisely define how the object is
 * set around the anchor point. For instance,<br>
 * <br>
 * <code>g.drawString("test",x,y,TOP|LEFT);</code><br>
 * <br>
 * draws a string and defines <code>(x,y)</code> as the upper left point of the text zone.<br>
 * <br>
 * <code>g.drawString("test",x,y,TOP|HCENTER);</code><br>
 * <br>
 * will draw string <code>"test"</code> above and centered on <code>(x,y)</code> .<br>
 * <br>
 * Note that any anchor constants combination must be limited to one of the horizontal constants (<code>LEFT</code>,
 * <code>HCENTER</code>, <code>RIGHT</code>) and one of the vertical constants (<code>TOP</code>, <code>BASELINE</code>
 * for text positioning exclusively, <code>VCENTER</code>, <code>BOTTOM</code>). The default anchor position, obtained
 * with value <code>0</code>, matches the <code>TOP | LEFT</code> constant combination.
 */
public class GraphicsContext {

	/**
	 * Constant for centering drawing horizontally around the anchor point.
	 * <p>
	 * Value <code>1</code> is assigned to <code>HCENTER</code>.
	 */
	public static final int HCENTER = 1;

	/**
	 * Constant for centering the drawing vertically around the anchor point.
	 * <p>
	 * Value <code>2</code> is assigned to <code>VCENTER</code>.
	 */
	public static final int VCENTER = 2;

	/**
	 * Constant for positioning the left side of the drawing at the anchor point.
	 * <p>
	 * Value <code>4</code> is assigned to <code>LEFT</code>.
	 */
	public static final int LEFT = 4;

	/**
	 * Constant for positioning the right side of the drawing at the anchor point.
	 * <p>
	 * Value <code>8</code> is assigned to <code>RIGHT</code>.
	 */
	public static final int RIGHT = 8;

	/**
	 * Constant for positioning the top of the drawing at the anchor point.
	 * <p>
	 * Value <code>16</code> is assigned to <code>TOP</code>.
	 */
	public static final int TOP = 16;

	/**
	 * Constant for positioning the bottom of the drawing at the anchor point.
	 * <p>
	 * Value <code>32</code> is assigned to <code>BOTTOM</code>.
	 */
	public static final int BOTTOM = 32;

	/**
	 * Constant for positioning the baseline of the text at the anchor point.
	 * <p>
	 * Value <code>64</code> is assigned to <code>BASELINE</code>.
	 */
	public static final int BASELINE = 64;

	/**
	 * Equivalent to <code>LEFT | TOP</code>.
	 */
	public static final int LEFT_TOP = LEFT | TOP;
	/**
	 * Equivalent to <code>LEFT | VCENTER</code>.
	 */
	public static final int LEFT_VCENTER = LEFT | VCENTER;
	/**
	 * Equivalent to <code>LEFT | BOTTOM</code>.
	 */
	public static final int LEFT_BOTTOM = LEFT | BOTTOM;

	/**
	 * Equivalent to <code>HCENTER | TOP</code>.
	 */
	public static final int HCENTER_TOP = HCENTER | TOP;
	/**
	 * Equivalent to <code>HCENTER | VCENTER</code>.
	 */
	public static final int HCENTER_VCENTER = HCENTER | VCENTER;
	/**
	 * Equivalent to <code>HCENTER | BOTTOM</code>.
	 */
	public static final int HCENTER_BOTTOM = HCENTER | BOTTOM;

	/**
	 * Equivalent to <code>RIGHT | TOP</code>.
	 */
	public static final int RIGHT_TOP = RIGHT | TOP;
	/**
	 * Equivalent to <code>RIGHT | VCENTER</code>.
	 */
	public static final int RIGHT_VCENTER = RIGHT | VCENTER;
	/**
	 * Equivalent to <code>RIGHT | BOTTOM</code>.
	 */
	public static final int RIGHT_BOTTOM = RIGHT | BOTTOM;

	/**
	 * Constant for the <code>SOLID</code> stroke style.
	 * <p>
	 * Value <code>0</code> is assigned to <code>SOLID</code>.
	 */
	public static final int SOLID = 0;

	/**
	 * Constant for the <code>DOTTED</code> stroke style.
	 * <p>
	 * Value <code>1</code> is assigned to <code>DOTTED</code>.
	 */
	public static final int DOTTED = 1;

	/**
	 * Maximal opacity.
	 *
	 * @see #drawImage(Image, int, int, int, int)
	 * @since 2.0
	 */
	public static final int OPAQUE = 0xff;

	/**
	 * Minimal opacity.
	 *
	 * @see #drawImage(Image, int, int, int, int)
	 * @since 2.0
	 */
	public static final int TRANSPARENT = 0x0;

	/**
	 * Forbidden constructor: use {@link Display#getNewGraphicsContext()} to get an instance of
	 * <code>GraphicsContext</code>.
	 *
	 * @see Display#getNewGraphicsContext()
	 */
	/* package */ GraphicsContext() {
		throw new RuntimeException();
	}

	/**
	 * Gets the alpha level for the given opacity.
	 * <p>
	 * It can be used to draw transparent images.
	 *
	 * @param opacityPercent
	 *            the expected opacity in percentage.
	 * @return the alpha level.
	 * @see #drawImage(Image, int, int, int, int)
	 * @since 2.0
	 */
	public static int getAlpha(int opacityPercent) {
		throw new RuntimeException();
	}

	/**
	 * Translates the <code>GraphicsContext</code> origin with the given vector <code>(x,y)</code>. Subsequent rendering
	 * operations on the graphics context will be relative to the new origin.
	 * <p>
	 * This method can be used to set an absolute origin to a <code>GraphicsContext</code>. For instance, the following
	 * code: <br>
	 * <code>g.translate(ax-g.getTranslateX(),ay-g.getTranslateY());</code><br>
	 * will set the origin of <code>g</code> at <code>(ax,ay)</code>.<br>
	 *
	 * @param x
	 *            the translation for the x coordinate.
	 * @param y
	 *            the translation for the y coordinate.
	 */
	public final void translate(int x, int y) {
		throw new RuntimeException();
	}

	/**
	 * Gets the x coordinate of the translated origin of the graphics context.
	 *
	 * @return x coordinate of the translated origin.
	 */
	public final int getTranslateX() {
		throw new RuntimeException();
	}

	/**
	 * Gets the y coordinate of the translated origin of the graphics context.
	 *
	 * @return y coordinate of the translated origin.
	 */
	public final int getTranslateY() {
		throw new RuntimeException();
	}

	/**
	 * Sets the current color.<br>
	 * Given value <code>rgbColor</code> is interpreted as a 24-bit RGB color, where the eight least significant bits
	 * matches the blue component, the next eight more significant bits matches the green component and the next eight
	 * more significant bits matches the red component.<br>
	 *
	 * @param rgbColor
	 *            the color to set.
	 */
	public final void setColor(int rgbColor) {
		throw new RuntimeException();
	}

	/**
	 * Sets the current background color.<br>
	 * Given value <code>rgbColor</code> is interpreted as a 24-bit RGB color, where the eight least significant bits
	 * matches the blue component, the next eight more significant bits matches the green component and the next eight
	 * more significant bits matches the red component.<br>
	 * The background color is used by several drawings: draw text, draw anti-aliased line etc. When not set, these
	 * algorithms have to read the pixel destination color before blending it with the current foreground color.<br>
	 * Note: this background color is useless for black and white displays.
	 *
	 * @param rgbColor
	 *            the color to set.
	 */
	public final void setBackgroundColor(int rgbColor) {
		throw new RuntimeException();
	}

	/**
	 * Removes the current background color.
	 */
	public final void removeBackgroundColor() {
		throw new RuntimeException();
	}

	/**
	 * Gets the current color: a 24-bits value interpreted as: <code>0xRRGGBB</code>, that is, the eight least
	 * significant bits give the blue color, the next eight bits the green value and the next eight bits the red color.
	 *
	 * @return the current color.
	 */
	public final int getColor() {
		throw new RuntimeException();
	}

	/**
	 * Gets the current background color.
	 *
	 * @return the current background color.
	 */
	public final int getBackgroundColor() {
		throw new RuntimeException();
	}

	/**
	 * Gets whether there us a background color.
	 *
	 * @return <code>true</code> if there is a background color, <code>false</code> otherwise.
	 */
	public final boolean hasBackgroundColor() {
		throw new RuntimeException();
	}

	/**
	 * Gets the color that will be displayed if the specified color is requested.<br>
	 * For example, with a monochrome display, this method will return either 0xFFFFFF (white) or 0x000000 (black)
	 * depending on the brightness of the specified color.
	 *
	 * @param color
	 *            the desired color in 0x00RRGGBB format.
	 * @return the corresponding color that will be displayed on the graphics context (in 0x00RRGGBB format).
	 */
	public final int getDisplayColor(int color) {
		throw new RuntimeException();
	}

	/**
	 * Sets the stroke style of this graphics context. It is used to draw lines, arcs, circle, rectangles and polygons.
	 *
	 * @param style
	 *            either {@link #SOLID} or {@link #DOTTED}.
	 * @throws IllegalArgumentException
	 *             if the style is not valid.
	 */
	public final void setStrokeStyle(int style) {
		throw new RuntimeException();
	}

	/**
	 * Gets the current stroke style: {@link #SOLID} or {@link #DOTTED}.
	 *
	 * @return the stroke style.
	 */
	public final int getStrokeStyle() {
		throw new RuntimeException();
	}

	/**
	 * Sets the font for subsequent text operations. If given font is <code>null</code>, the
	 * <code>GraphicsContext</code>'s font is set to <code>DisplayFont.getDefaultFont()</code>.
	 *
	 * @param font
	 *            the new font to use.
	 */
	public void setFont(Font font) {
		throw new RuntimeException();
	}

	/**
	 * Gets the current font.
	 *
	 * @return the current font.
	 */
	public final Font getFont() {
		throw new RuntimeException();
	}

	/**
	 * Sets the clipping area to be the intersection of the specified rectangle with the current clipping rectangle. It
	 * is legal to specify a clip rectangle whose width or height is zero or negative. In this case the clip is
	 * considered to be empty, that is, no pixels are contained within it. Therefore, if any graphics operations are
	 * issued under such a clip, no pixels will be modified.
	 *
	 * @param x
	 *            the x coordinate of the rectangle.
	 * @param y
	 *            the y coordinate of the rectangle.
	 * @param width
	 *            the width of the rectangle.
	 * @param height
	 *            the height of the rectangle.
	 */
	public final void clipRect(int x, int y, int width, int height) {
		throw new RuntimeException();
	}

	/**
	 * Sets the current clipping zone to the rectangle defined by the given location <code>(x,y)</code> and size
	 * <code>(width,height)</code>. Given width or height may be zero or negative, in that case the clip is considered
	 * to be empty, i.e. it contains no pixels. Nothing is done when drawing in an empty clip. Rendering operations have
	 * no effect outside of the clipping area.
	 *
	 * @param x
	 *            the x coordinate of the new clip rectangle.
	 * @param y
	 *            the y coordinate of the new clip rectangle.
	 * @param width
	 *            the width of the new clip rectangle.
	 * @param height
	 *            the height of the new clip rectangle.
	 */
	public final void setClip(int x, int y, int width, int height) {
		throw new RuntimeException();
	}

	/**
	 * Gets the x offset of the current clipping zone, relative to the graphics context's origin.
	 *
	 * @return the x offset of the current clipping zone.
	 */
	public final int getClipX() {
		throw new RuntimeException();
	}

	/**
	 * Gets the y offset of the current clipping zone, relative to graphics context's origin.
	 *
	 * @return the y offset of the current clipping zone.
	 */
	public final int getClipY() {
		throw new RuntimeException();
	}

	/**
	 * Gets the width of the current clipping zone.
	 *
	 * @return the width of the current clipping zone.
	 */
	public final int getClipWidth() {
		throw new RuntimeException();
	}

	/**
	 * Gets the height of the current clipping zone.
	 *
	 * @return the height of the current clipping zone.
	 */
	public final int getClipHeight() {
		throw new RuntimeException();
	}

	/**
	 * Draws a pixel at <code>(x,y)</code> using the current color.
	 *
	 * @param x
	 *            the x coordinate of the pixel.
	 * @param y
	 *            the y coordinate of the pixel.
	 */
	public final void drawPixel(int x, int y) {
		throw new RuntimeException();
	}

	/**
	 * Obtains the ARGB color of the pixel at <code>(x,y)</code>. The read color may be different than the drawing
	 * color. It is screen dependent, according to the number of bits per pixels (see {@link Display#getBPP()}).
	 *
	 * @param x
	 *            the x coordinate of the pixel.
	 * @param y
	 *            the y coordinate of the pixel.
	 * @return the ARGB color of the pixel.
	 * @throws IllegalArgumentException
	 *             if the pixel coordinates are out of bounds of the source graphics context.
	 */
	public final int readPixel(int x, int y) {
		throw new RuntimeException();
	}

	/**
	 * Draws an horizontal line from <code>(x,y)</code> to <code>(x+width,y)</code> using the current color and stroke
	 * style. The drawn line counts <code>(width+1)</code> pixels.<br>
	 * If <code>width</code> is negative, nothing is drawn.
	 *
	 * @param x
	 *            the x coordinate of the start of the line.
	 * @param y
	 *            the y coordinate of the start of the line.
	 * @param width
	 *            the width of the horizontal line to draw.
	 */
	public final void drawHorizontalLine(int x, int y, int width) {
		throw new RuntimeException();
	}

	/**
	 * Draws a vertical line from <code>(x,y)</code> to <code>(x,y+height-1)</code> using the current color and stroke
	 * style. The drawn line counts <code>(height+1)</code> pixels.<br>
	 * If <code>height</code> is negative, nothing is drawn.
	 *
	 * @param x
	 *            the x coordinate of the start of the line.
	 * @param y
	 *            the y coordinate of the start of the line.
	 * @param height
	 *            the width of the vertical line to draw.
	 */
	public final void drawVerticalLine(int x, int y, int height) {
		throw new RuntimeException();
	}

	/**
	 * Draws a line from <code>(x1,y1)</code> to <code>(x2,y2)</code> using the current color and stroke style.
	 *
	 * @param x1
	 *            the x coordinate of the start of the line.
	 * @param y1
	 *            the y coordinate of the start of the line.
	 * @param x2
	 *            the x coordinate of the end of the line.
	 * @param y2
	 *            the y coordinate of the end of the line.
	 */
	public final void drawLine(int x1, int y1, int x2, int y2) {
		throw new RuntimeException();
	}

	/**
	 * Draws the outline of the specified rectangle using the current color and stroke style. The drawn rectangle
	 * includes <code>(width+1)*(height+1)</code> pixels.<br>
	 * If either <code>width</code> or <code>height</code> is negative, nothing is drawn.
	 *
	 * @param x
	 *            the x coordinate of the rectangle to draw.
	 * @param y
	 *            the y coordinate of the rectangle to draw.
	 * @param width
	 *            the width of the rectangle to draw.
	 * @param height
	 *            the height of the rectangle to draw.
	 */
	public final void drawRect(int x, int y, int width, int height) {
		throw new RuntimeException();
	}

	/**
	 * Fills the specified rectangle with the current color. If either <code>width</code> or <code>height</code> is
	 * negative or zero, nothing is drawn.
	 *
	 * @param x
	 *            the x coordinate of the rectangle to be filled.
	 * @param y
	 *            the y coordinate of the rectangle to be filled.
	 * @param width
	 *            the width of the rectangle to be filled.
	 * @param height
	 *            the height of the rectangle to be filled.
	 */
	public final void fillRect(int x, int y, int width, int height) {
		throw new RuntimeException();
	}

	/**
	 * Draws the outline of the specified rounded corner rectangle using the current color and stroke style. Drawn
	 * rectangle is <code>width+1</code> -pixel wide and <code>height+1</code>-pixel high. If either <code>width</code>
	 * or <code>height</code> is negative, nothing is drawn.
	 *
	 * @param x
	 *            the x coordinate of the rectangle to draw.
	 * @param y
	 *            the y coordinate of the rectangle to draw.
	 * @param width
	 *            the width of the rectangle to draw.
	 * @param height
	 *            the height of the rectangle to draw.
	 * @param arcWidth
	 *            the horizontal diameter of the arc at the corners.
	 * @param arcHeight
	 *            the vertical diameter of the arc at the corners.
	 */
	public final void drawRoundRect(int x, int y, int width, int height, int arcWidth, int arcHeight) {
		throw new RuntimeException();
	}

	/**
	 * Fills the specified rounded corner rectangle with the current color. If either <code>width</code> or
	 * <code>height</code> is negative or zero, nothing is drawn.
	 *
	 * @param x
	 *            the x coordinate of the rectangle to fill.
	 * @param y
	 *            the y coordinate of the rectangle to fill.
	 * @param width
	 *            the width of the rectangle to fill.
	 * @param height
	 *            the height of the rectangle to fill.
	 * @param arcWidth
	 *            the horizontal diameter of the arc at the corners.
	 * @param arcHeight
	 *            the vertical diameter of the arc at the corners.
	 */
	public final void fillRoundRect(int x, int y, int width, int height, int arcWidth, int arcHeight) {
		throw new RuntimeException();
	}

	/**
	 * Draws the closed polygon which is defined by the array of integer coordinates, using the current color and stroke
	 * style. Lines are drawn between each consecutive pair, and between the first pair and last pair in the array.
	 * <p>
	 *
	 * @param xys
	 *            the array of coordinates : x1,y1,......xn,yn.
	 * @throws NullPointerException
	 *             if the given array is <code>null</code>.
	 * @throws IllegalArgumentException
	 *             if the given array length is odd.
	 */
	public final void drawPolygon(int[] xys) {
		throw new RuntimeException();
	}

	/**
	 * Draws the closed polygon which is defined by the array of integer coordinates, using the current color and stroke
	 * style. Lines are drawn between each consecutive pair, and between the first pair and last pair in the array.
	 *
	 * @param xys
	 *            the array of coordinates : x1,y1,......xn,yn.
	 * @param offset
	 *            the <code>x1</code> index in <code>xys</code>.
	 * @param length
	 *            the number of coordinates, must be even.
	 * @throws NullPointerException
	 *             if the given array is <code>null</code>.
	 * @throws IllegalArgumentException
	 *             if the given array length is odd.
	 * @throws ArrayIndexOutOfBoundsException
	 *             the wanted data is outside the array bounds.
	 */
	public final void drawPolygon(int[] xys, int offset, int length) {
		throw new RuntimeException();
	}

	/**
	 * Fills the closed polygon which is defined by the array of integer coordinates, using the current color. Lines are
	 * drawn between each consecutive pair, and between the first pair and last pair in the array. The lines connecting
	 * each pair of points are included in the filled polygon. The effect is identical to<br>
	 * <code>fillPolygon(xys,0,xys.length);</code>
	 *
	 * @param xys
	 *            the array of coordinates : x1,y1,......xn,yn.
	 * @throws NullPointerException
	 *             if the given array is <code>null</code>.
	 * @throws IllegalArgumentException
	 *             if the given array length is odd.
	 */
	public final void fillPolygon(int[] xys) {
		throw new RuntimeException();
	}

	/**
	 * Fills the closed polygon which is defined by the array of integer coordinates, using the current color. Lines are
	 * drawn between each consecutive pair, and between the first pair and last pair in the array. The lines connecting
	 * each pair of points are included in the filled polygon. The effect is identical to<br>
	 * <code>fillPolygon(xys,0,xys.length);</code>
	 *
	 * @param xys
	 *            the array of coordinates : x1,y1,......xn,yn.
	 * @param offset
	 *            the <code>x1</code> index in <code>xys</code>.
	 * @param length
	 *            the number of coordinates, must be even.
	 * @throws ArrayIndexOutOfBoundsException
	 *             if <code>offset</code> and <code>length</code> do not specify a valid range within <code>xys</code>.
	 * @throws NullPointerException
	 *             if the <code>xys</code> array is null.
	 * @throws IllegalArgumentException
	 *             if the <code>xys</code> length is odd.
	 */
	public final void fillPolygon(int[] xys, int offset, int length) {
		throw new RuntimeException();
	}

	/**
	 * Draws the outline of a circular arc covering the specified square, using the current color and stroke style.
	 * <p>
	 * The arc is drawn from <code>startAngle</code> up to <code>arcAngle</code> degrees. The center of the arc is
	 * defined as the center of the rectangle whose origin is at <code>(x,y)</code> (upper-left corner) and whose
	 * dimension is given by <code>diameter</code>.
	 * <p>
	 * Angles are interpreted such that 0 degrees is at the 3 o'clock position. A positive value indicates a
	 * counter-clockwise rotation while a negative value indicates a clockwise rotation.
	 * <p>
	 * If the given diameter is negative, nothing is drawn.
	 * <p>
	 * The angles are given relative to the rectangle. For instance an angle of 45 degrees is always defined by the line
	 * from the center of the rectangle to the upper right corner of the rectangle. Thus for a non squarred rectangle
	 * angles are skewed along either height or width.
	 *
	 * @param x
	 *            the x coordinate of the upper-left corner of the rectangle where the arc is drawn
	 * @param y
	 *            the y coordinate of the upper-left corner of the rectangle where the arc is drawn
	 * @param diameter
	 *            the diameter of the arc to draw
	 * @param startAngle
	 *            the beginning angle of the arc to draw
	 * @param arcAngle
	 *            the angular extent of the arc from <code>startAngle</code>
	 */
	public final void drawCircleArc(int x, int y, int diameter, int startAngle, int arcAngle) {
		throw new RuntimeException();
	}

	/**
	 * Draws the outline of a elliptical arc covering the specified rectangle, using the current color and stroke style.
	 * <p>
	 * The arc is drawn from <code>startAngle</code> up to <code>arcAngle</code> degrees. The center of the arc is
	 * defined as the center of the rectangle whose origin is at <code>(x,y)</code> (upper-left corner) and whose
	 * dimension is given by <code>width</code> and <code>height</code>.
	 * <p>
	 * Angles are interpreted such that 0 degrees is at the 3 o'clock position. A positive value indicates a
	 * counter-clockwise rotation while a negative value indicates a clockwise rotation.
	 * <p>
	 * If either the given width or height is negative, nothing is drawn.
	 * <p>
	 * The angles are given relative to the rectangle. For instance an angle of 45 degrees is always defined by the line
	 * from the center of the rectangle to the upper right corner of the rectangle. Thus for a non squarred rectangle
	 * angles are skewed along either height or width.
	 *
	 * @param x
	 *            the x coordinate of the upper-left corner of the rectangle where the arc is drawn
	 * @param y
	 *            the y coordinate of the upper-left corner of the rectangle where the arc is drawn
	 * @param width
	 *            the width of the arc to draw
	 * @param height
	 *            the height of the arc to draw
	 * @param startAngle
	 *            the beginning angle of the arc to draw
	 * @param arcAngle
	 *            the angular extent of the arc from <code>startAngle</code>
	 */
	public final void drawEllipseArc(int x, int y, int width, int height, int startAngle, int arcAngle) {
		throw new RuntimeException();
	}

	/**
	 * Fills a circular arc covering the specified square with the current color.
	 * <p>
	 * The arc is drawn from <code>startAngle</code> up to <code>arcAngle</code> degrees. The center of the arc is
	 * defined as the center of the rectangle whose origin is at <code>(x,y)</code> (upper-left corner) and whose
	 * dimension is given by <code>diameter</code>.
	 * <p>
	 * Angles are interpreted such that 0 degrees is at the 3 o'clock position. A positive value indicates a
	 * counter-clockwise rotation while a negative value indicates a clockwise rotation.
	 * <p>
	 * This method fills the area bounded from the center of the arc to the arc itself.
	 * <p>
	 * If the given diameter is negative, nothing is drawn.
	 * <p>
	 * The angles are given relatively to the rectangle. That is to say that the angle of 45 degrees is always defined
	 * by the line from the center of the rectangle to the upper-right corner of the rectangle. Thus for a non squarred
	 * rectangle angles are skewed along either height or width.
	 * <p>
	 *
	 * @param x
	 *            the x coordinate of the upper-left corner of the rectangle where the arc is filled.
	 * @param y
	 *            the y coordinate of the upper-left corner of the rectangle where the arc is filled.
	 * @param diameter
	 *            the diameter of the arc to fill
	 * @param startAngle
	 *            the beginning angle of the arc to draw
	 * @param arcAngle
	 *            the angular extent of the arc from <code>startAngle</code>
	 */
	public final void fillCircleArc(int x, int y, int diameter, int startAngle, int arcAngle) {
		throw new RuntimeException();
	}

	/**
	 * Fills an elliptical arc covering the specified rectangle with the current color.
	 * <p>
	 * The arc is drawn from <code>startAngle</code> up to <code>arcAngle</code> degrees. The center of the arc is
	 * defined as the center of the rectangle whose origin is at <code>(x,y)</code> (upper-left corner) and whose
	 * dimension is given by <code>width</code> and <code>height</code>.
	 * <p>
	 * Angles are interpreted such that 0 degrees is at the 3 o'clock position. A positive value indicates a
	 * counter-clockwise rotation while a negative value indicates a clockwise rotation.
	 * <p>
	 * This method fills the area bounded from the center of the arc to the arc itself.
	 * <p>
	 * If either the given width or height is negative, nothing is drawn.
	 * <p>
	 * The angles are given relatively to the rectangle. That is to say that the angle of 45 degrees is always defined
	 * by the line from the center of the rectangle to the upper-right corner of the rectangle. Thus for a non squared
	 * rectangle angles are skewed along either height or width.
	 *
	 * @param x
	 *            the x coordinate of the upper-left corner of the rectangle where the arc is filled.
	 * @param y
	 *            the y coordinate of the upper-left corner of the rectangle where the arc is filled.
	 * @param width
	 *            the width of the arc to fill.
	 * @param height
	 *            the height of the arc to fill.
	 * @param startAngle
	 *            the beginning angle of the arc to draw.
	 * @param arcAngle
	 *            the angular extent of the arc from the given start angle.
	 */
	public final void fillEllipseArc(int x, int y, int width, int height, int startAngle, int arcAngle) {
		throw new RuntimeException();
	}

	/**
	 * Draws the outline of a circle covering the rectangle specified by its diameter, using the current color and
	 * stroke style.
	 * <p>
	 * The center of the circle is defined as the center of the rectangle whose origin is at the given coordinates
	 * (upper-left corner) and whose dimension is given by the diameter parameter.
	 * <p>
	 * If the given diameter is negative, nothing is drawn.
	 *
	 * @param x
	 *            the x coordinate of the upper-left corner of the rectangle where the circle is drawn.
	 * @param y
	 *            the y coordinate of the upper-left corner of the rectangle where the circle is drawn.
	 * @param diameter
	 *            the diameter of the circle to draw.
	 */
	public final void drawCircle(int x, int y, int diameter) {
		throw new RuntimeException();
	}

	/**
	 * Fills a circle covering the rectangle specified by its diameter with the current color.
	 * <p>
	 * The center of the circle is defined as the center of the rectangle whose origin is at the given coordinates
	 * (upper-left corner) and whose dimension is given by the diameter parameter.
	 * <p>
	 * If the given diameter is negative, nothing is drawn.
	 *
	 * @param x
	 *            the x coordinate of the upper-left corner of the rectangle where the circle is filled.
	 * @param y
	 *            the y coordinate of the upper-left corner of the rectangle where the circle is filled.
	 * @param diameter
	 *            the diameter of the circle to fill.
	 */
	public final void fillCircle(int x, int y, int diameter) {
		throw new RuntimeException();
	}

	/**
	 * Draws the outline of a ellipse covering the specified rectangle, using the current color and stroke style.
	 * <p>
	 * The center of the ellipse is defined as the center of the rectangle whose origin is at the given coordinates
	 * (upper-left corner) and whose dimension is given by the width and height parameters.
	 * <p>
	 * If either the given width or height is negative, nothing is drawn.
	 *
	 * @param x
	 *            the x coordinate of the upper-left corner of the rectangle where the ellipse is drawn.
	 * @param y
	 *            the y coordinate of the upper-left corner of the rectangle where the ellipse is drawn.
	 * @param width
	 *            the width of the ellipse to draw.
	 * @param height
	 *            the height of the ellipse to draw.
	 */
	public final void drawEllipse(int x, int y, int width, int height) {
		throw new RuntimeException();
	}

	/**
	 * Fills a ellipse covering the specified rectangle with the current color.
	 * <p>
	 * The center of the ellipse is defined as the center of the rectangle whose origin is at the given coordinates
	 * (upper-left corner) and whose dimension is given by the width and height parameters.
	 * <p>
	 * If either the given width or height is negative, nothing is drawn.
	 *
	 * @param x
	 *            the x coordinate of the upper-left corner of the rectangle where the ellipse is filled.
	 * @param y
	 *            the y coordinate of the upper-left corner of the rectangle where the ellipse is filled.
	 * @param width
	 *            the width of the ellipse to fill.
	 * @param height
	 *            the height of the ellipse to fill.
	 */
	public final void fillEllipse(int x, int y, int width, int height) {
		throw new RuntimeException();
	}

	/**
	 * Draws an image at the given anchor point.
	 * <p>
	 * The image anchor point is at the given position. Position constants may be given to specify the precise location
	 * of the image around the anchor point.
	 * <p>
	 * Equivalent to calling {@link #drawImage(Image, int, int, int, int)} with {@link #OPAQUE} as alpha.
	 *
	 * @param img
	 *            the image to draw.
	 * @param x
	 *            the x coordinate of the anchor point.
	 * @param y
	 *            the y coordinate of the anchor point.
	 * @param anchor
	 *            position of the image around the anchor point.
	 * @throws IllegalArgumentException
	 *             if the given anchor is not a valid value ({@link #BASELINE} is illegal).
	 * @throws NullPointerException
	 *             if the given image is <code>null</code>.
	 */
	public final void drawImage(Image img, int x, int y, int anchor) {
		drawImage(img, x, y, anchor, OPAQUE);
	}

	/**
	 * Draws an image at the given anchor point. In addition with {@link #drawImage(Image, int, int, int)}, this method
	 * allows to specify the global opacity value to apply during the image rendering.
	 *
	 * @param img
	 *            the image to draw.
	 * @param x
	 *            the x coordinate of the anchor point.
	 * @param y
	 *            the y coordinate of the anchor point.
	 * @param anchor
	 *            position of the image around the anchor point.
	 * @param alpha
	 *            the global opacity rendering value.
	 * @throws IllegalArgumentException
	 *             if the given anchor is not a valid value ({@link #BASELINE} is illegal).
	 * @throws IllegalArgumentException
	 *             if the given alpha is not a value between {@link #TRANSPARENT} and {@link #OPAQUE}.
	 * @throws NullPointerException
	 *             if the given image is <code>null</code>.
	 * @since 2.0
	 */
	public final void drawImage(Image img, int x, int y, int anchor, int alpha) {
		throw new RuntimeException();
	}

	/**
	 * Draws a region of an image.
	 * <p>
	 * Equivalent to calling {@link #drawRegion(Image, int, int, int, int, int, int, int, int)} with {@link #OPAQUE} as
	 * alpha.
	 *
	 * @param src
	 *            the image to copy from.
	 * @param xSrc
	 *            the x coordinate of the upper-left corner of the region to copy.
	 * @param ySrc
	 *            the y coordinate of the upper-left corner of the region to copy.
	 * @param width
	 *            the width of the region to copy.
	 * @param height
	 *            the height of the region to copy.
	 * @param xDest
	 *            the x coordinate of the anchor point in the destination.
	 * @param yDest
	 *            the y coordinate of the anchor point in the destination.
	 * @param anchor
	 *            the position of the region around the anchor point.
	 * @throws NullPointerException
	 *             if the given image is <code>null</code>.
	 * @throws IllegalArgumentException
	 *             if the given anchor is not a valid value ({@link #BASELINE} is illegal).
	 */
	public final void drawRegion(Image src, int xSrc, int ySrc, int width, int height, int xDest, int yDest,
			int anchor) {
		throw new RuntimeException();
	}

	/**
	 * Draws a region of an image.
	 * <p>
	 * The region of the image to draw is given relative to the image (origin at the upper-left corner) as a rectangle.
	 * <p>
	 * The image region anchor point in destination is at the given relative position. Position constants may be given
	 * to specify the precise location of the image around the anchor point.
	 * <p>
	 * If the specified source region exceeds the image bounds, the copied region is limited to the image boundary. If
	 * the copied region goes out of the bounds of the graphics context area, pixels out of the range will not be drawn.
	 * <p>
	 * This method allows to specify the opacity value to apply during the image rendering.
	 *
	 * @param src
	 *            the image to copy from.
	 * @param xSrc
	 *            the x coordinate of the upper-left corner of the region to copy.
	 * @param ySrc
	 *            the y coordinate of the upper-left corner of the region to copy.
	 * @param width
	 *            the width of the region to copy.
	 * @param height
	 *            the height of the region to copy.
	 * @param xDest
	 *            the x coordinate of the anchor point in the destination.
	 * @param yDest
	 *            the y coordinate of the anchor point in the destination.
	 * @param anchor
	 *            the position of the region around the anchor point.
	 * @param alpha
	 *            the alpha to apply to the region.
	 * @throws NullPointerException
	 *             if the given image is <code>null</code>.
	 * @throws IllegalArgumentException
	 *             if the given anchor is not a valid value ({@link #BASELINE} is illegal).
	 * @throws IllegalArgumentException
	 *             if the given alpha is not a value between {@link #TRANSPARENT} and {@link #OPAQUE}.
	 * @since 2.0
	 */
	public final void drawRegion(Image src, int xSrc, int ySrc, int width, int height, int xDest, int yDest, int anchor,
			int alpha) {
		throw new RuntimeException();
	}

	/**
	 * Draws a region of a display.
	 * <p>
	 * Equivalent to calling {@link #drawRegion(Display, int, int, int, int, int, int, int, int)} with {@link #OPAQUE}
	 * as alpha.
	 *
	 * @param display
	 *            the display to copy from.
	 * @param xSrc
	 *            the x coordinate of the upper-left corner of the region to copy.
	 * @param ySrc
	 *            the y coordinate of the upper-left corner of the region to copy.
	 * @param width
	 *            the width of the region to copy.
	 * @param height
	 *            the height of the region to copy.
	 * @param xDest
	 *            the x coordinate of the anchor point in the destination.
	 * @param yDest
	 *            the y coordinate of the anchor point in the destination.
	 * @param anchor
	 *            the position of the region around the anchor point.
	 * @throws NullPointerException
	 *             if the given display is <code>null</code>.
	 * @throws IllegalArgumentException
	 *             if the given anchor is not a valid value ({@link #BASELINE} is illegal).
	 * @see Display#getScreenshot()
	 * @see Display#getScreenshot(int, int, int, int)
	 */
	public void drawRegion(Display display, int xSrc, int ySrc, int width, int height, int xDest, int yDest,
			int anchor) {
		throw new RuntimeException();
	}

	/**
	 * Draws a region of a display.
	 * <p>
	 * The region of the display to draw is given relative to the display (origin at the upper-left corner) as a
	 * rectangle.
	 * <p>
	 * If the specified source region exceeds the image bounds, the copied region is limited to the image boundary. If
	 * the copied region goes out of the bounds of the graphics context area, pixels out of the range will not be drawn.
	 * <p>
	 * This method allows to specify the opacity value to apply during the display content rendering.
	 *
	 * @param display
	 *            the display to copy from.
	 * @param xSrc
	 *            the x coordinate of the upper-left corner of the region to copy.
	 * @param ySrc
	 *            the y coordinate of the upper-left corner of the region to copy.
	 * @param width
	 *            the width of the region to copy.
	 * @param height
	 *            the height of the region to copy.
	 * @param xDest
	 *            the x coordinate of the anchor point in the destination.
	 * @param yDest
	 *            the y coordinate of the anchor point in the destination.
	 * @param anchor
	 *            the position of the region around the anchor point.
	 * @param alpha
	 *            the alpha to apply to the region.
	 * @throws NullPointerException
	 *             if the given display is <code>null</code>.
	 * @throws IllegalArgumentException
	 *             if the given anchor is not a valid value ({@link #BASELINE} is illegal).
	 * @throws IllegalArgumentException
	 *             if the given alpha is not a value between {@link #TRANSPARENT} and {@link #OPAQUE}.
	 * @see Display#getScreenshot()
	 * @see Display#getScreenshot(int, int, int, int)
	 * @since 2.0
	 */
	public void drawRegion(Display display, int xSrc, int ySrc, int width, int height, int xDest, int yDest, int anchor,
			int alpha) {
		throw new RuntimeException();
	}

	/**
	 * Obtains ARGB pixel data from the specified region of this graphics context and stores it in the provided array of
	 * integers. Each pixel value is stored in @{code 0xAARRGGBB} format, where the high-order byte contains the alpha
	 * channel and the remaining bytes contain color components for red, green and blue, respectively. The alpha channel
	 * specifies the opacity of the pixel, where a value of @{code 0x00} represents a pixel that is fully transparent
	 * and a value of @{code 0xFF} represents a fully opaque pixel.
	 * <p>
	 * Color values may be resampled to reflect the display capabilities of the device (for example, red, green or blue
	 * pixels may all be represented by the same gray value on a grayscale device).
	 * <p>
	 * The scan length specifies the relative offset within the array between the corresponding pixels of consecutive
	 * rows. In order to prevent rows of stored pixels from overlapping, the absolute value of scan length must be
	 * greater than or equal to the given width. Negative values of scan length are allowed. In all cases, this must
	 * result in every reference being within the bounds of the ARGB pixel data array.
	 *
	 * @param argbData
	 *            an array of integers in which the ARGB pixel data is stored.
	 * @param offset
	 *            the index into the array where the first ARGB value is stored.
	 * @param scanlength
	 *            the relative offset in the array between corresponding pixels in consecutive rows of the region.
	 * @param x
	 *            the x-coordinate of the upper left corner of the region.
	 * @param y
	 *            the y-coordinate of the upper left corner of the region.
	 * @param width
	 *            the width of the region.
	 * @param height
	 *            the height of the region.
	 *
	 * @throws ArrayIndexOutOfBoundsException
	 *             if the requested operation would attempt to access an element in the given ARGB pixel data array
	 *             whose index is either negative or beyond its length (the contents of the array are unchanged).
	 * @throws IllegalArgumentException
	 *             if the area being retrieved exceeds the bounds of the source graphics context.
	 * @throws IllegalArgumentException
	 *             if the absolute value of the given scan length is less than the given width.
	 * @throws NullPointerException
	 *             if the given ARGB pixel data is @{code null}.
	 */
	public final void getARGB(int[] argbData, int offset, int scanlength, int x, int y, int width, int height) {
		throw new RuntimeException();
	}

	/**
	 * Draws the string using the current font and color.<br>
	 * The text anchor point is at position <code>(x,y)</code>. Position constants may be given to specify the precise
	 * location of the text around the anchor point.
	 * <p>
	 * See {@link GraphicsContext} for details of anchors.
	 *
	 * @param str
	 *            the string to draw.
	 * @param x
	 *            the x coordinate of the anchor point.
	 * @param y
	 *            the y coordinate of the anchor point.
	 * @param anchor
	 *            position of the text around the anchor point.
	 * @throws NullPointerException
	 *             if the given string is <code>null</code>.
	 * @throws IllegalArgumentException
	 *             if anchor is not a valid value.
	 */
	public final void drawString(String str, int x, int y, int anchor) {
		throw new RuntimeException();
	}

	/**
	 * Draws the string from <code>offset</code> to <code>offset+length</code> using the current font and color.
	 * <p>
	 * The text anchor point is at position <code>(x,y)</code>. Position constants may be given to specify the precise
	 * location of the text around the anchor point.
	 * <p>
	 * See {@link GraphicsContext} for details of anchors.
	 *
	 * @param str
	 *            the string to draw.
	 * @param offset
	 *            index of the first character in the string to draw.
	 * @param len
	 *            number of characters to draw from the given offset.
	 * @param x
	 *            the x coordinate of the anchor point.
	 * @param y
	 *            the y coordinate of the anchor point.
	 * @param anchor
	 *            position of the string text around the anchor point.
	 * @throws StringIndexOutOfBoundsException
	 *             if the given offset and length do not specify a valid range within the given string.
	 * @throws IllegalArgumentException
	 *             if anchor is not a valid value.
	 * @throws NullPointerException
	 *             if the given string is <code>null</code>.
	 */
	public final void drawSubstring(String str, int offset, int len, int x, int y, int anchor) {
		throw new RuntimeException();
	}

	/**
	 * Draws a character using the current font and color.<br>
	 * The text anchor point is at position <code>(x,y)</code>. Position constants may be given to specify the precise
	 * location of the character around the anchor point.
	 *
	 * @param character
	 *            the character to draw.
	 * @param x
	 *            the x coordinate of the anchor point.
	 * @param y
	 *            the y coordinate of the anchor point.
	 * @param anchor
	 *            position of the character around the anchor point.
	 * @throws IllegalArgumentException
	 *             if the given anchor is not a valid value.
	 */
	public final void drawChar(char character, int x, int y, int anchor) {
		throw new RuntimeException();
	}

	/**
	 * Draws some characters using the current font and color.<br>
	 * The text anchor point is at position <code>(x,y)</code>. Position constants may be given to specify the precise
	 * location of the text around the anchor point.
	 *
	 * @param data
	 *            the array of characters to draw.
	 * @param offset
	 *            offset of the first character to draw in the char array.
	 * @param length
	 *            the number of characters to draw from the offset.
	 * @param x
	 *            the x coordinate of the anchor point.
	 * @param y
	 *            the y coordinate of the anchor point.
	 * @param anchor
	 *            position of the text around the anchor point.
	 * @throws IndexOutOfBoundsException
	 *             if the given offset and length do not specify a valid range within the given char array.
	 * @throws IllegalArgumentException
	 *             if the given anchor is not a valid value.
	 * @throws NullPointerException
	 *             if the given char array is <code>null</code>.
	 */
	public final void drawChars(char[] data, int offset, int length, int x, int y, int anchor) {
		throw new RuntimeException();
	}

	/**
	 * Enables (disables) truncation when rendering characters. When enabled, a text that would be outside the current
	 * clip will have its last visible character replaced by ellipsis (three dots).
	 *
	 * @param enable
	 *            <code>true</code> to enable the ellipsis mode, <code>false</code> otherwise.
	 */
	public void setEllipsis(boolean enable) {
		throw new RuntimeException();
	}

	/**
	 * Gets the display associated with the GraphicsContext.
	 *
	 * @return the display associated with the GraphicsContext
	 */
	public Display getDisplay() {
		throw new RuntimeException();
	}

	/**
	 * Gets whether the truncation mechanism is enabled or not.
	 *
	 * @return <code>true</code> if the truncation mechanism is enabled, <code>false</code> otherwise.
	 */
	public boolean getEllipsis() {
		throw new RuntimeException();
	}

}