/*
 * Java
 *
 * Copyright 2021-2023 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package com.microej.microvg.test;

import static org.junit.Assert.assertEquals;

import org.junit.AfterClass;
import org.junit.Assert;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;

import ej.bon.Constants;
import ej.microui.display.Colors;
import ej.microui.display.Display;
import ej.microui.display.GraphicsContext;
import ej.microui.display.Painter;
import ej.microvg.Matrix;
import ej.microvg.Path;
import ej.microvg.VectorGraphicsPainter;
import ej.microvg.VectorImage;

/**
 * Tests the drawing of colored path.
 */
public class TestPath {

	private static final float PATH_BOUNDS_TOLERANCE = 1.0E-5f;

	/**
	 * Starts MicroUI.
	 */
	@BeforeClass
	public static void pre() {
		TestUtilities.startMicroUI();
	}

	/**
	 * Stops MicroUI.
	 */
	@AfterClass
	public static void post() {
		TestUtilities.stopMicroUI();
	}

	/**
	 * Resets the content of the screen to black.
	 */
	@Before
	public static void preTest() {
		TestUtilities.clearScreen();
	}

	/**
	 * Tests a square path without initial move.
	 */
	@Test
	public static void testPathNoMove() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		Path path = new Path();
		path.lineTo(100, 0);
		path.lineTo(100, 100);
		path.lineTo(0, 100);
		path.close();

		g.setColor(0xff0000);
		VectorGraphicsPainter.fillPath(g, path, new Matrix());
		display.requestFlush();

		TestUtilities.check("top left", 0, 0, g, 0xff0000); //$NON-NLS-1$
		TestUtilities.check("center", 49, 49, g, 0xff0000); //$NON-NLS-1$
		TestUtilities.check("bottom right", 99, 99, g, 0xff0000); //$NON-NLS-1$
	}

	/**
	 * Tests filling a square path with x,y parameters.
	 */
	@Test
	public static void testPathXY() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		Path path = new Path();
		path.lineTo(100, 0);
		path.lineTo(100, 100);
		path.lineTo(0, 100);
		path.close();

		g.setColor(0xff0000);
		VectorGraphicsPainter.fillPath(g, path, 20, 40);
		display.requestFlush();

		TestUtilities.check("top left", 20, 40, g, 0xff0000); //$NON-NLS-1$
		TestUtilities.check("center", 69, 89, g, 0xff0000); //$NON-NLS-1$
		TestUtilities.check("bottom right", 119, 139, g, 0xff0000); //$NON-NLS-1$
	}

	/**
	 * Tests a square path without initial move with a different origin.
	 */
	@Test
	public static void testPathNoMoveOrigin() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		Path path = new Path();
		path.setOrigin(50, 50);
		path.lineTo(100, 0);
		path.lineTo(100, 100);
		path.lineTo(0, 100);
		path.close();

		g.setColor(0xff0000);
		VectorGraphicsPainter.fillPath(g, path, new Matrix());
		display.requestFlush();

		TestUtilities.check("top left", 50, 50, g, 0xff0000); //$NON-NLS-1$
		TestUtilities.check("center", 99, 99, g, 0xff0000); //$NON-NLS-1$
		TestUtilities.check("bottom right", 149, 149, g, 0xff0000); //$NON-NLS-1$
	}

	/**
	 * Tests a square path without close.
	 */
	@Test
	public static void testPathNoClose() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		Path path = new Path();
		path.moveTo(0, 0);
		path.lineTo(100, 0);
		path.lineTo(100, 100);
		path.lineTo(0, 100);

		g.setColor(0xff0000);
		VectorGraphicsPainter.fillPath(g, path, new Matrix());
		display.requestFlush();

		TestUtilities.check("top left", 0, 0, g, 0xff0000); //$NON-NLS-1$
		TestUtilities.check("center", 49, 49, g, 0xff0000); //$NON-NLS-1$
		TestUtilities.check("bottom right", 99, 99, g, 0xff0000); //$NON-NLS-1$
	}

	/**
	 * Tests resetting the origin.
	 */
	@Test
	public static void testPathResetOrigin() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		Path path = new Path();
		path.setOrigin(100, 100);
		path.moveTo(0, 0);
		path.lineTo(100, 0);
		path.reset();
		path.moveTo(50, 50);
		path.lineTo(150, 50);
		path.lineTo(150, 150);
		path.lineTo(50, 150);
		path.close();

		g.setColor(0xff0000);
		VectorGraphicsPainter.fillPath(g, path, new Matrix());
		display.requestFlush();

		TestUtilities.check("top left", 50, 50, g, 0xff0000); //$NON-NLS-1$
		TestUtilities.check("center", 99, 99, g, 0xff0000); //$NON-NLS-1$
		TestUtilities.check("bottom right", 149, 149, g, 0xff0000); //$NON-NLS-1$
	}

	/**
	 * Tests relative move.
	 */
	@Test
	public static void testPathMoveRelative() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		Path path = new Path();
		path.moveTo(25, 25);
		path.moveToRelative(25, 25);
		path.lineTo(150, 50);
		path.lineTo(150, 150);
		path.lineTo(50, 150);
		path.close();

		g.setColor(0xff0000);
		VectorGraphicsPainter.fillPath(g, path, new Matrix());
		display.requestFlush();

		TestUtilities.check("top left", 50, 50, g, 0xff0000); //$NON-NLS-1$
		TestUtilities.check("center", 99, 99, g, 0xff0000); //$NON-NLS-1$
		TestUtilities.check("bottom right", 149, 149, g, 0xff0000); //$NON-NLS-1$
	}

	/**
	 * Tests path with an inner move (two square shapes).
	 */
	@Test
	public static void testPathInnerMove() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		Path path = new Path();
		path.moveTo(0, 0);
		path.lineTo(100, 0);
		path.lineTo(100, 100);
		path.lineTo(0, 100);

		path.moveTo(150, 150);
		path.lineTo(250, 150);
		path.lineTo(250, 250);
		path.lineTo(150, 250);
		path.close();

		g.setColor(0xff0000);
		VectorGraphicsPainter.fillPath(g, path, new Matrix());
		display.requestFlush();

		TestUtilities.check("top left", 0, 0, g, 0xff0000); //$NON-NLS-1$
		TestUtilities.check("center", 49, 49, g, 0xff0000); //$NON-NLS-1$
		TestUtilities.check("bottom right", 99, 99, g, 0xff0000); //$NON-NLS-1$
		TestUtilities.check("top left", 150, 150, g, 0xff0000); //$NON-NLS-1$
		TestUtilities.check("center", 199, 199, g, 0xff0000); //$NON-NLS-1$
		TestUtilities.check("bottom right", 249, 249, g, 0xff0000); //$NON-NLS-1$
	}

	/**
	 * Tests path with an inner move (two square shapes) with an intermediate close.
	 */
	@Test
	public static void testPathInnerMoveClose() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		Path path = new Path();
		path.moveTo(0, 0);
		path.lineTo(100, 0);
		path.lineTo(100, 100);
		path.lineTo(0, 100);
		path.close();

		path.moveTo(150, 150);
		path.lineTo(250, 150);
		path.lineTo(250, 250);
		path.lineTo(150, 250);
		path.close();

		g.setColor(0xff0000);
		VectorGraphicsPainter.fillPath(g, path, new Matrix());
		display.requestFlush();

		TestUtilities.check("top left", 0, 0, g, 0xff0000); //$NON-NLS-1$
		TestUtilities.check("center", 49, 49, g, 0xff0000); //$NON-NLS-1$
		TestUtilities.check("bottom right", 99, 99, g, 0xff0000); //$NON-NLS-1$
		TestUtilities.check("top left", 150, 150, g, 0xff0000); //$NON-NLS-1$
		TestUtilities.check("center", 199, 199, g, 0xff0000); //$NON-NLS-1$
		TestUtilities.check("bottom right", 249, 249, g, 0xff0000); //$NON-NLS-1$
	}

	/**
	 * Tests a square path.
	 */
	@Test
	public static void testPathLine() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		Path path = new Path();
		path.moveTo(50, 50);
		path.lineTo(150, 50);
		path.lineTo(150, 150);
		path.lineTo(50, 150);
		path.close();

		g.setColor(0xff0000);
		VectorGraphicsPainter.fillPath(g, path, new Matrix());
		display.requestFlush();

		TestUtilities.check("top left", 50, 50, g, 0xff0000); //$NON-NLS-1$
		TestUtilities.check("center", 99, 99, g, 0xff0000); //$NON-NLS-1$
		TestUtilities.check("bottom right", 149, 149, g, 0xff0000); //$NON-NLS-1$
	}

	/**
	 * Tests a square path with relative operations without initial move.
	 */
	@Test
	public static void testPathLineRelativeNoMove() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		Path path = new Path();
		path.lineToRelative(100, 0);
		path.lineToRelative(0, 100);
		path.lineToRelative(-100, 0);
		path.close();

		g.setColor(0xff0000);
		VectorGraphicsPainter.fillPath(g, path, new Matrix());
		display.requestFlush();

		TestUtilities.check("top left", 0, 0, g, 0xff0000); //$NON-NLS-1$
		TestUtilities.check("center", 49, 49, g, 0xff0000); //$NON-NLS-1$
		TestUtilities.check("bottom right", 99, 99, g, 0xff0000); //$NON-NLS-1$
	}

	/**
	 * Tests a square path with relative operations.
	 */
	@Test
	public static void testPathLineRelative() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		Path path = new Path();
		path.moveTo(50, 50);
		path.lineToRelative(100, 0);
		path.lineToRelative(0, 100);
		path.lineToRelative(-100, 0);
		path.close();

		g.setColor(0xff0000);
		VectorGraphicsPainter.fillPath(g, path, new Matrix());
		display.requestFlush();

		TestUtilities.check("top left", 50, 50, g, 0xff0000); //$NON-NLS-1$
		TestUtilities.check("center", 99, 99, g, 0xff0000); //$NON-NLS-1$
		TestUtilities.check("bottom right", 149, 149, g, 0xff0000); //$NON-NLS-1$
	}

	/**
	 * Tests a square path with a different origin.
	 */
	@Test
	public static void testPathLineOrigin() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		Path path = new Path();
		path.setOrigin(50, 50);
		path.moveTo(0, 0);
		path.lineTo(100, 0);
		path.lineTo(100, 100);
		path.lineTo(0, 100);
		path.close();

		g.setColor(0xff0000);
		VectorGraphicsPainter.fillPath(g, path, new Matrix());
		display.requestFlush();

		TestUtilities.check("top left", 50, 50, g, 0xff0000); //$NON-NLS-1$
		TestUtilities.check("center", 99, 99, g, 0xff0000); //$NON-NLS-1$
		TestUtilities.check("bottom right", 149, 149, g, 0xff0000); //$NON-NLS-1$
	}

	/**
	 * Tests a square path with a different origin and relative operations.
	 */
	@Test
	public static void testPathLineOriginRelative() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		Path path = new Path();
		path.setOrigin(50, 50);
		path.moveTo(0, 0);
		path.lineToRelative(100, 0);
		path.lineToRelative(0, 100);
		path.lineToRelative(-100, 0);
		path.close();

		g.setColor(0xff0000);
		VectorGraphicsPainter.fillPath(g, path, new Matrix());
		display.requestFlush();

		TestUtilities.check("top left", 50, 50, g, 0xff0000); //$NON-NLS-1$
		TestUtilities.check("center", 99, 99, g, 0xff0000); //$NON-NLS-1$
		TestUtilities.check("bottom right", 149, 149, g, 0xff0000); //$NON-NLS-1$
	}

	/**
	 * Tests a path with quad bezier (looks like batman logo) without initial move.
	 */
	@Test
	public static void testPathQuadNoMove() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		Path path = new Path();
		path.quadTo(50, 50, 100, 0);
		path.quadTo(150, 50, 100, 100);
		path.quadTo(50, 50, 0, 100);
		path.quadTo(-50, 50, 0, 0);

		g.setColor(0xff0000);
		VectorGraphicsPainter.fillPath(g, path, new Matrix());
		display.requestFlush();

		TestUtilities.check("top left", 0, 1, g, 0xff0000); //$NON-NLS-1$
		TestUtilities.check("center", 49, 49, g, 0xff0000); //$NON-NLS-1$
		TestUtilities.check("bottom right", 99, 98, g, 0xff0000); //$NON-NLS-1$
		TestUtilities.check("top", 50, 0, g, 0x000000); //$NON-NLS-1$
		TestUtilities.check("bottom", 50, 99, g, 0x000000); //$NON-NLS-1$
		TestUtilities.check("left", 0, 49, g, 0xff0000); //$NON-NLS-1$
		TestUtilities.check("right", 120, 49, g, 0xff0000); //$NON-NLS-1$
	}

	/**
	 * Tests a path with quad bezier (looks like batman logo).
	 */
	@Test
	public static void testPathQuad() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		Path path = new Path();
		path.moveTo(50, 50);
		path.quadTo(100, 100, 150, 50);
		path.quadTo(200, 100, 150, 150);
		path.quadTo(100, 100, 50, 150);
		path.quadTo(0, 100, 50, 50);

		g.setColor(0xff0000);
		VectorGraphicsPainter.fillPath(g, path, new Matrix());
		display.requestFlush();

		TestUtilities.check("top left", 50, 51, g, 0xff0000); //$NON-NLS-1$
		TestUtilities.check("center", 99, 99, g, 0xff0000); //$NON-NLS-1$
		TestUtilities.check("bottom right", 149, 148, g, 0xff0000); //$NON-NLS-1$
		TestUtilities.check("top", 100, 50, g, 0x000000); //$NON-NLS-1$
		TestUtilities.check("bottom", 100, 149, g, 0x000000); //$NON-NLS-1$
		TestUtilities.check("left", 30, 99, g, 0xff0000); //$NON-NLS-1$
		TestUtilities.check("right", 170, 99, g, 0xff0000); //$NON-NLS-1$
	}

	/**
	 * Tests a path with quad bezier (looks like batman logo) with relative operations without initial move.
	 */
	@Test
	public static void testPathQuadRelativeNoMove() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		Path path = new Path();
		path.quadToRelative(50, 50, 100, 0);
		path.quadToRelative(50, 50, 0, 100);
		path.quadToRelative(-50, -50, -100, 0);
		path.quadToRelative(-50, -50, 0, -100);

		g.setColor(0xff0000);
		VectorGraphicsPainter.fillPath(g, path, new Matrix());
		display.requestFlush();

		TestUtilities.check("top left", 0, 1, g, 0xff0000); //$NON-NLS-1$
		TestUtilities.check("center", 49, 49, g, 0xff0000); //$NON-NLS-1$
		TestUtilities.check("bottom right", 99, 98, g, 0xff0000); //$NON-NLS-1$
		TestUtilities.check("top", 50, 0, g, 0x000000); //$NON-NLS-1$
		TestUtilities.check("bottom", 50, 99, g, 0x000000); //$NON-NLS-1$
		TestUtilities.check("left", 0, 49, g, 0xff0000); //$NON-NLS-1$
		TestUtilities.check("right", 120, 49, g, 0xff0000); //$NON-NLS-1$
	}

	/**
	 * Tests a path with quad bezier (looks like batman logo) with relative operations.
	 */
	@Test
	public static void testPathQuadRelative() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		Path path = new Path();
		path.moveTo(50, 50);
		path.quadToRelative(50, 50, 100, 0);
		path.quadToRelative(50, 50, 0, 100);
		path.quadToRelative(-50, -50, -100, 0);
		path.quadToRelative(-50, -50, 0, -100);

		g.setColor(0xff0000);
		VectorGraphicsPainter.fillPath(g, path, new Matrix());
		display.requestFlush();

		TestUtilities.check("top left", 50, 51, g, 0xff0000); //$NON-NLS-1$
		TestUtilities.check("center", 99, 99, g, 0xff0000); //$NON-NLS-1$
		TestUtilities.check("bottom right", 149, 148, g, 0xff0000); //$NON-NLS-1$
		TestUtilities.check("top", 100, 50, g, 0x000000); //$NON-NLS-1$
		TestUtilities.check("bottom", 100, 149, g, 0x000000); //$NON-NLS-1$
		TestUtilities.check("left", 30, 99, g, 0xff0000); //$NON-NLS-1$
		TestUtilities.check("right", 170, 99, g, 0xff0000); //$NON-NLS-1$
	}

	/**
	 * Tests a path with quad bezier (looks like batman logo) with a different origin.
	 */
	@Test
	public static void testPathQuadOrigin() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		Path path = new Path();
		path.setOrigin(50, 50);
		path.moveTo(0, 0);
		path.quadTo(50, 50, 100, 0);
		path.quadTo(150, 50, 100, 100);
		path.quadTo(50, 50, 0, 100);
		path.quadTo(-50, 50, 0, 0);

		g.setColor(0xff0000);
		VectorGraphicsPainter.fillPath(g, path, new Matrix());
		display.requestFlush();

		TestUtilities.check("top left", 50, 51, g, 0xff0000); //$NON-NLS-1$
		TestUtilities.check("center", 99, 99, g, 0xff0000); //$NON-NLS-1$
		TestUtilities.check("bottom right", 149, 148, g, 0xff0000); //$NON-NLS-1$
		TestUtilities.check("top", 100, 50, g, 0x000000); //$NON-NLS-1$
		TestUtilities.check("bottom", 100, 149, g, 0x000000); //$NON-NLS-1$
		TestUtilities.check("left", 30, 99, g, 0xff0000); //$NON-NLS-1$
		TestUtilities.check("right", 170, 99, g, 0xff0000); //$NON-NLS-1$
	}

	/**
	 * Tests a path with quad bezier (looks like batman logo) with a different origin and relative operations.
	 */
	@Test
	public static void testPathQuadOriginRelative() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		Path path = new Path();
		path.setOrigin(50, 50);
		path.moveTo(0, 0);
		path.quadToRelative(50, 50, 100, 0);
		path.quadToRelative(50, 50, 0, 100);
		path.quadToRelative(-50, -50, -100, 0);
		path.quadToRelative(-50, -50, 0, -100);

		g.setColor(0xff0000);
		VectorGraphicsPainter.fillPath(g, path, new Matrix());
		display.requestFlush();

		TestUtilities.check("top left", 50, 51, g, 0xff0000); //$NON-NLS-1$
		TestUtilities.check("center", 99, 99, g, 0xff0000); //$NON-NLS-1$
		TestUtilities.check("bottom right", 149, 148, g, 0xff0000); //$NON-NLS-1$
		TestUtilities.check("top", 100, 50, g, 0x000000); //$NON-NLS-1$
		TestUtilities.check("bottom", 100, 149, g, 0x000000); //$NON-NLS-1$
		TestUtilities.check("left", 30, 99, g, 0xff0000); //$NON-NLS-1$
		TestUtilities.check("right", 170, 99, g, 0xff0000); //$NON-NLS-1$
	}

	/**
	 * Tests a path with cubic bezier (looks like batman logo) without initial move.
	 */
	@Test
	public static void testPathCubicNoMove() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		Path path = new Path();
		path.cubicTo(25, 50, 75, 50, 100, 0);
		path.cubicTo(150, 25, 150, 75, 100, 100);
		path.cubicTo(75, 50, 25, 50, 0, 100);
		path.cubicTo(-50, 75, -50, 25, 0, 0);

		g.setColor(0xff0000);
		VectorGraphicsPainter.fillPath(g, path, new Matrix());
		display.requestFlush();

		TestUtilities.check("top left", 0, 2, g, 0xff0000); //$NON-NLS-1$
		TestUtilities.check("center", 49, 49, g, 0xff0000); //$NON-NLS-1$
		TestUtilities.check("bottom right", 99, 97, g, 0xff0000); //$NON-NLS-1$
		TestUtilities.check("top", 50, 0, g, 0x000000); //$NON-NLS-1$
		TestUtilities.check("bottom", 50, 99, g, 0x000000); //$NON-NLS-1$
		TestUtilities.check("left", 0, 49, g, 0xff0000); //$NON-NLS-1$
		TestUtilities.check("right", 120, 49, g, 0xff0000); //$NON-NLS-1$
	}

	/**
	 * Tests a path with cubic bezier (looks like batman logo).
	 */
	@Test
	public static void testPathCubic() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		Path path = new Path();
		path.moveTo(50, 50);
		path.cubicTo(75, 100, 125, 100, 150, 50);
		path.cubicTo(200, 75, 200, 125, 150, 150);
		path.cubicTo(125, 100, 75, 100, 50, 150);
		path.cubicTo(0, 125, 0, 75, 50, 50);

		g.setColor(0xff0000);
		VectorGraphicsPainter.fillPath(g, path, new Matrix());
		display.requestFlush();

		TestUtilities.check("top left", 50, 52, g, 0xff0000); //$NON-NLS-1$
		TestUtilities.check("center", 99, 99, g, 0xff0000); //$NON-NLS-1$
		TestUtilities.check("bottom right", 149, 147, g, 0xff0000); //$NON-NLS-1$
		TestUtilities.check("top", 100, 50, g, 0x000000); //$NON-NLS-1$
		TestUtilities.check("bottom", 100, 149, g, 0x000000); //$NON-NLS-1$
		TestUtilities.check("left", 30, 99, g, 0xff0000); //$NON-NLS-1$
		TestUtilities.check("right", 170, 99, g, 0xff0000); //$NON-NLS-1$
	}

	/**
	 * Tests a path with cubic bezier (looks like batman logo) with relative operations without initial move.
	 */
	@Test
	public static void testPathCubicRelativeNoMove() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		Path path = new Path();
		path.cubicToRelative(25, 50, 75, 50, 100, 0);
		path.cubicToRelative(50, 25, 50, 75, 0, 100);
		path.cubicToRelative(-25, -50, -75, -50, -100, 0);
		path.cubicToRelative(-50, -25, -50, -75, 0, -100);

		g.setColor(0xff0000);
		VectorGraphicsPainter.fillPath(g, path, new Matrix());
		display.requestFlush();

		TestUtilities.check("top left", 0, 2, g, 0xff0000); //$NON-NLS-1$
		TestUtilities.check("center", 49, 49, g, 0xff0000); //$NON-NLS-1$
		TestUtilities.check("bottom right", 99, 97, g, 0xff0000); //$NON-NLS-1$
		TestUtilities.check("top", 50, 0, g, 0x000000); //$NON-NLS-1$
		TestUtilities.check("bottom", 50, 99, g, 0x000000); //$NON-NLS-1$
		TestUtilities.check("left", 0, 49, g, 0xff0000); //$NON-NLS-1$
		TestUtilities.check("right", 120, 49, g, 0xff0000); //$NON-NLS-1$
	}

	/**
	 * Tests a path with cubic bezier (looks like batman logo) with relative operations.
	 */
	@Test
	public static void testPathCubicRelative() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		Path path = new Path();
		path.moveTo(50, 50);
		path.cubicToRelative(25, 50, 75, 50, 100, 0);
		path.cubicToRelative(50, 25, 50, 75, 0, 100);
		path.cubicToRelative(-25, -50, -75, -50, -100, 0);
		path.cubicToRelative(-50, -25, -50, -75, 0, -100);

		g.setColor(0xff0000);
		VectorGraphicsPainter.fillPath(g, path, new Matrix());
		display.requestFlush();

		TestUtilities.check("top left", 50, 52, g, 0xff0000); //$NON-NLS-1$
		TestUtilities.check("center", 99, 99, g, 0xff0000); //$NON-NLS-1$
		TestUtilities.check("bottom right", 149, 147, g, 0xff0000); //$NON-NLS-1$
		TestUtilities.check("top", 100, 50, g, 0x000000); //$NON-NLS-1$
		TestUtilities.check("bottom", 100, 149, g, 0x000000); //$NON-NLS-1$
		TestUtilities.check("left", 30, 99, g, 0xff0000); //$NON-NLS-1$
		TestUtilities.check("right", 170, 99, g, 0xff0000); //$NON-NLS-1$
	}

	/**
	 * Tests a path with cubic bezier (looks like batman logo) with a different origin.
	 */
	@Test
	public static void testPathCubicOrigin() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		Path path = new Path();
		path.setOrigin(50, 50);
		path.moveTo(0, 0);
		path.cubicTo(25, 50, 75, 50, 100, 0);
		path.cubicTo(150, 25, 150, 75, 100, 100);
		path.cubicTo(75, 50, 25, 50, 0, 100);
		path.cubicTo(-50, 75, -50, 25, 0, 0);

		g.setColor(0xff0000);
		VectorGraphicsPainter.fillPath(g, path, new Matrix());
		display.requestFlush();

		TestUtilities.check("top left", 50, 52, g, 0xff0000); //$NON-NLS-1$
		TestUtilities.check("center", 99, 99, g, 0xff0000); //$NON-NLS-1$
		TestUtilities.check("bottom right", 149, 147, g, 0xff0000); //$NON-NLS-1$
		TestUtilities.check("top", 100, 50, g, 0x000000); //$NON-NLS-1$
		TestUtilities.check("bottom", 100, 149, g, 0x000000); //$NON-NLS-1$
		TestUtilities.check("left", 30, 99, g, 0xff0000); //$NON-NLS-1$
		TestUtilities.check("right", 170, 99, g, 0xff0000); //$NON-NLS-1$
	}

	/**
	 * Tests a path with cubic bezier (looks like batman logo) with a different origin and relative operations.
	 */
	@Test
	public static void testPathCubicOriginRelative() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		Path path = new Path();
		path.setOrigin(50, 50);
		path.moveTo(0, 0);
		path.cubicToRelative(25, 50, 75, 50, 100, 0);
		path.cubicToRelative(50, 25, 50, 75, 0, 100);
		path.cubicToRelative(-25, -50, -75, -50, -100, 0);
		path.cubicToRelative(-50, -25, -50, -75, 0, -100);

		g.setColor(0xff0000);
		VectorGraphicsPainter.fillPath(g, path, new Matrix());
		display.requestFlush();

		TestUtilities.check("top left", 50, 52, g, 0xff0000); //$NON-NLS-1$
		TestUtilities.check("center", 99, 99, g, 0xff0000); //$NON-NLS-1$
		TestUtilities.check("bottom right", 149, 147, g, 0xff0000); //$NON-NLS-1$
		TestUtilities.check("top", 100, 50, g, 0x000000); //$NON-NLS-1$
		TestUtilities.check("bottom", 100, 149, g, 0x000000); //$NON-NLS-1$
		TestUtilities.check("left", 30, 99, g, 0xff0000); //$NON-NLS-1$
		TestUtilities.check("right", 170, 99, g, 0xff0000); //$NON-NLS-1$
	}

	/**
	 * Tests a path with negative bounds, then translated.
	 */
	@Test
	public static void testPathNegativeBounds() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		Path path = new Path();
		path.moveTo(-100, -100);
		path.lineToRelative(0, -100);
		path.lineToRelative(-100, 0);
		path.lineToRelative(0, 100);
		path.close();

		Matrix m = new Matrix();
		m.setTranslate(300, 300); // Brings top left to 100,100

		g.setColor(0xff0000);
		VectorGraphicsPainter.fillPath(g, path, m);
		display.requestFlush();

		TestUtilities.check("top left", 102, 102, g, 0xff0000); //$NON-NLS-1$
		TestUtilities.check("center", 150, 150, g, 0xff0000); //$NON-NLS-1$
		TestUtilities.check("bottom right", 198, 198, g, 0xff0000); //$NON-NLS-1$
		TestUtilities.check("top", 150, 102, g, 0xff0000); //$NON-NLS-1$
		TestUtilities.check("bottom", 150, 198, g, 0xff0000); //$NON-NLS-1$
		TestUtilities.check("left", 102, 150, g, 0xff0000); //$NON-NLS-1$
		TestUtilities.check("right", 198, 150, g, 0xff0000); //$NON-NLS-1$
	}

	/**
	 * Tests a path with clipping.
	 */
	@Test
	public static void testPathWithClip() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		int x = 100;
		int y = 100;
		int width = 60;
		int height = 60;
		int testThickness = 1;

		int errors = 0;

		Path path = new Path();
		path.moveTo(0, 0);
		path.lineToRelative(0, height);
		path.lineToRelative(width, 0);
		path.lineToRelative(0, -height);
		path.close();

		Matrix m = new Matrix();
		m.setTranslate(x, y); // Brings top left to x,y

		// Clip larger than path
		int outterClip = 10;
		g.setClip(x - outterClip, y - outterClip, width + 2 * outterClip, height + 2 * outterClip);

		g.setColor(0x0000ff);
		VectorGraphicsPainter.fillPath(g, path, m);

		// Reset clip
		g.setClip(0, 0, display.getWidth(), display.getHeight());
		display.requestFlush();

		errors += TestUtilities.checkPeripheralArea("clip over inside", 0x0000ff, x + testThickness, y + testThickness,
				width - 2 * testThickness, height - 2 * testThickness, testThickness, 0, false);
		assertEquals(0, errors);
		errors += TestUtilities.checkPeripheralArea("clip over outside", 0x000000, x, y, width, height, testThickness,
				0, false);
		assertEquals(0, errors);

		// Clip fit path
		x += 100;
		m.postTranslate(100, 0);

		outterClip = 0;
		g.setClip(x - outterClip, y - outterClip, width + 2 * outterClip, height + 2 * outterClip);

		g.setColor(0x0000ff);
		VectorGraphicsPainter.fillPath(g, path, m);

		// Reset clip
		g.setClip(0, 0, display.getWidth(), display.getHeight());

		display.requestFlush();

		errors += TestUtilities.checkPeripheralArea("clip fit inside", 0x0000ff, x + testThickness, y + testThickness,
				width - 2 * testThickness, height - 2 * testThickness, testThickness, 0, false);
		assertEquals(0, errors);
		errors += TestUtilities.checkPeripheralArea("clip fit outside", 0x000000, x, y, width, height, testThickness, 0,
				false);
		assertEquals(0, errors);
		// Clip thinner than path
		x += 100;
		m.postTranslate(100, 0);

		outterClip = -10;
		g.setClip(x - outterClip, y - outterClip, width + 2 * outterClip, height + 2 * outterClip);

		g.setColor(0x0000ff);
		VectorGraphicsPainter.fillPath(g, path, m);

		// Reset clip
		g.setClip(0, 0, display.getWidth(), display.getHeight());

		display.requestFlush();
		errors += TestUtilities.checkPeripheralArea("clip inner inside", 0x0000ff, x - outterClip + testThickness,
				y - outterClip + testThickness, width + 2 * outterClip - 2 * testThickness,
				height + 2 * outterClip - 2 * testThickness, testThickness, 0, false);
		assertEquals(0, errors);
		errors += TestUtilities.checkPeripheralArea("clip inner outside", 0x000000, x - outterClip, y - outterClip,
				width + 2 * outterClip, height + 2 * outterClip, testThickness, 0, false);
		assertEquals(0, errors);
	}

	/**
	 * Task M0092MEJAUI-2753: a very long path is drawn in several passes by the GPU.
	 *
	 * Test success if there is no deadlock.
	 */
	@Test
	public static void testVeryLongPath() {
		// Skip this test on Android because Android's path parser throws an IllegalArgumentException (STRING_TOO_LARGE)
		// when the path data is too long.
		if (TestUtilities.isOnAndroid()) {
			return;
		}

		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();
		final VectorImage image = VectorImage.getImage("/com/microej/microvg/test/very_long_path.xml");

		float imageWidth = image.getWidth();
		float imageHeight = image.getHeight();

		final Matrix matrix = new Matrix();

		for (int i = 1; i < 400; i++) {

			if (Constants.getBoolean(TestUtilities.DEBUG_CONSTANT)) {
				System.out.println("i:" + i);
			}

			g.setColor(Colors.BLACK);
			Painter.fillRectangle(g, 0, 0, i, i);

			matrix.setScale(i / imageWidth, i / imageHeight);
			VectorGraphicsPainter.drawImage(g, image, matrix);

			display.flush();
		}

	}

	/**
	 * Tests the bounds of a path with no point.
	 */
	@Test
	public void testBoundsEmpty() {
		Path path = new Path();
		assertBounds(path, 0.0f, 0.0f, 0.0f, 0.0f);
	}

	/**
	 * Tests the bounds of a path with a single point.
	 */
	@Test
	public void testBoundsSinglePoint() {
		Path path = new Path();
		path.moveTo(4.0f, 5.0f);
		assertBounds(path, 4.0f, 4.0f, 5.0f, 5.0f);
	}

	/**
	 * Tests the bounds of a square path.
	 */
	@Test
	public void testBoundsSquare() {
		Path path = new Path();
		path.moveTo(1.0f, 1.5f);
		path.lineTo(2.0f, 1.5f);
		path.lineTo(2.0f, 2.5f);
		path.lineTo(1.0f, 2.5f);
		assertBounds(path, 1.0f, 2.0f, 1.5f, 2.5f);
	}

	/**
	 * Tests the bounds of a square path with several moves.
	 */
	@Test
	public void testBoundsSquareSeveralMoves() {
		Path path = new Path();
		path.moveTo(1.0f, 1.5f);
		path.moveTo(2.0f, 1.5f);
		path.moveTo(2.0f, 2.5f);
		path.moveTo(1.0f, 2.5f);
		assertBounds(path, 1.0f, 2.0f, 1.5f, 2.5f);
	}

	/**
	 * Tests the bounds of a square path with relative coordinates.
	 */
	@Test
	public void testBoundsSquareRelative() {
		Path path = new Path();
		path.moveToRelative(1.0f, 1.5f);
		path.lineToRelative(1.0f, 0.0f);
		path.lineToRelative(0.0f, 1.0f);
		path.lineToRelative(-1.0f, 0.0f);
		assertBounds(path, 1.0f, 2.0f, 1.5f, 2.5f);
	}

	/**
	 * Tests the bounds of a path with 4 points and straight edges.
	 */
	@Test
	public void testBoundsNegative() {
		Path path = new Path();
		path.moveTo(1.0f, 2.0f);
		path.lineTo(4.0f, -1.0f);
		path.lineTo(5.0f, 6.0f);
		path.lineTo(-2.0f, 5.0f);
		assertBounds(path, -2.0f, 5.0f, -1.0f, 6.0f);
	}

	/**
	 * Tests the bounds of a path with 4 points and quad edges.
	 */
	@Test
	public void testBoundsQuad() {
		Path path = new Path();
		path.moveTo(1.0f, 1.5f);
		path.quadTo(1.5f, 1.0f, 2.0f, 1.5f);
		path.quadTo(2.5f, 2.0f, 2.0f, 2.5f);
		path.quadTo(1.5f, 3.0f, 1.0f, 2.5f);
		path.quadTo(0.5f, 2.0f, 1.0f, 1.5f);
		assertBounds(path, 0.5f, 2.5f, 1.0f, 3.0f);
	}

	/**
	 * Tests the bounds of a path with 4 points and quad edges with relative coordinates.
	 */
	@Test
	public void testBoundsQuadRelative() {
		Path path = new Path();
		path.moveToRelative(1.0f, 1.5f);
		path.quadToRelative(0.5f, -0.5f, 1.0f, 0.0f);
		path.quadToRelative(0.5f, 0.5f, 0.0f, 1.0f);
		path.quadToRelative(-0.5f, 0.5f, -1.0f, 0.0f);
		path.quadToRelative(-0.5f, -0.5f, 0.0f, -1.0f);
		assertBounds(path, 0.5f, 2.5f, 1.0f, 3.0f);
	}

	/**
	 * Tests the bounds of a path with 4 points and cubic edges.
	 */
	@Test
	public void testBoundsCirc() {
		Path path = new Path();
		path.moveTo(1.0f, 1.5f);
		path.cubicTo(1.3f, 0.8f, 1.7f, 1.2f, 2.0f, 1.5f);
		path.cubicTo(2.3f, 1.8f, 2.7f, 2.2f, 2.0f, 2.5f);
		path.cubicTo(1.7f, 3.2f, 1.3f, 2.8f, 1.0f, 2.5f);
		path.cubicTo(0.7f, 2.2f, 0.3f, 1.8f, 1.0f, 1.5f);
		assertBounds(path, 0.3f, 2.7f, 0.8f, 3.2f);
	}

	/**
	 * Tests the bounds of a path with 4 points and cubic edges with relative coordinates.
	 */
	@Test
	public void testBoundsCircRelative() {
		Path path = new Path();
		path.moveToRelative(1.0f, 1.5f);
		path.cubicToRelative(0.3f, -0.7f, 0.7f, -0.3f, 1.0f, 0.0f);
		path.cubicToRelative(0.3f, 0.3f, 0.7f, 0.7f, 0.0f, 1.0f);
		path.cubicToRelative(-0.3f, 0.7f, -0.7f, 0.3f, -1.0f, 0.0f);
		path.cubicToRelative(-0.3f, -0.3f, -0.7f, -0.7f, 0.0f, -1.0f);
		assertBounds(path, 0.3f, 2.7f, 0.8f, 3.2f);
	}

	/**
	 * Tests the bounds of a reset path.
	 */
	@Test
	public void testBoundsReset() {
		Path path = new Path();
		path.moveTo(1.0f, 2.0f);
		path.lineTo(4.0f, -1.0f);
		path.lineTo(5.0f, 6.0f);
		path.lineTo(-2.0f, 5.0f);
		path.reset();
		assertBounds(path, 0.0f, 0.0f, 0.0f, 0.0f);
		path.moveTo(4.0f, 5.5f);
		path.lineTo(5.0f, 5.5f);
		path.lineTo(5.0f, 6.5f);
		path.lineTo(4.0f, 6.5f);
		assertBounds(path, 4.0f, 5.0f, 5.5f, 6.5f);
	}

	private static void assertBounds(Path path, float left, float right, float top, float bottom) {
		Assert.assertEquals("left", left, path.getLeftBound(), PATH_BOUNDS_TOLERANCE);
		Assert.assertEquals("right", right, path.getRightBound(), PATH_BOUNDS_TOLERANCE);
		Assert.assertEquals("top", top, path.getTopBound(), PATH_BOUNDS_TOLERANCE);
		Assert.assertEquals("bottom", bottom, path.getBottomBound(), PATH_BOUNDS_TOLERANCE);
	}
}
