/*
 * Java
 *
 * Copyright 2009-2019 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.mwt;

import java.util.Iterator;
import java.util.NoSuchElementException;

import com.is2t.tools.ArrayTools;

import ej.annotation.NonNull;
import ej.annotation.Nullable;
import ej.microui.display.GraphicsContext;
import ej.microui.event.generator.Command;

/**
 * A composite is a widget that can contain other {@link Widget} instances, following the composite pattern.
 * <p>
 * The children are stored in a list. The order of the list defines the front-to-back stacking order of the widgets
 * within the composite. The first widget in the list is at the back of the stacking order.
 * <p>
 * A widget cannot be added two times in a hierarchy.
 */
public abstract class Composite extends Widget {

	@NonNull
	private static final Widget[] EMPTY_WIDGETS = new Widget[0];

	@NonNull
	private Widget[] widgets;

	/**
	 * Creates a new composite.<br>
	 * Its bounds will be set to <code>0</code>.
	 */
	public Composite() {
		super();
		this.widgets = EMPTY_WIDGETS;
	}

	/**
	 * Adds the specified widget to the end of the list of children of this composite.
	 * <p>
	 * If the composite is on a panel hierarchy, it is invalidated.
	 *
	 * @param widget
	 *            the widget to add.
	 * @throws NullPointerException
	 *             if the specified widget is <code>null</code>.
	 * @throws IllegalArgumentException
	 *             if the specified widget is already in a hierarchy (already contained in a composite or panel).
	 * @see #invalidate()
	 * @see Panel#setWidget(Widget)
	 */
	protected void add(@NonNull Widget widget) throws NullPointerException, IllegalArgumentException {
		synchronized (this) {
			if (widget.panel != null || widget.parent != null) {
				throw new IllegalArgumentException();
			}
			widget.setParent(this);
			this.widgets = (Widget[]) ArrayTools.add(this.widgets, widget);
		}
		// the panel needs to be laid out
		invalidate();
	}

	/**
	 * Removes the specified widget from the list of children of this composite.
	 * <p>
	 * If the composite is on a panel hierarchy, it is invalidated.
	 * <p>
	 * If the widget is not in the list of children of the composite, nothing is done.
	 *
	 * @param widget
	 *            the widget to remove
	 * @throws NullPointerException
	 *             if the specified widget is null
	 * @see #invalidate()
	 */
	protected void remove(@NonNull Widget widget) throws NullPointerException {
		synchronized (this) {
			int oldLength = this.widgets.length;
			this.widgets = (Widget[]) ArrayTools.remove(this.widgets, widget);
			if (oldLength != this.widgets.length) {
				// Check focus before freeing the widget.
				boolean ownsFocus = widget.isFocusOwner();
				// Free the widget before changing the focus.
				widget.setParent(null);
				Panel panel = this.panel;
				if (panel != null && ownsFocus) {
					// the child owns the panel focus
					panel.setFocus(null);
				}
			}
		}
		// the panel needs to be laid out
		invalidate();
	}

	/**
	 * Removes all the widgets from the list of children of this composite.
	 * <p>
	 * If the composite is on a panel hierarchy, it is invalidated.
	 *
	 * @see #invalidate()
	 */
	protected void removeAllWidgets() {
		Panel panel = this.panel;
		// Check focus before freeing the widgets.
		boolean focusOwner = isFocusOwner();
		// Free the widgets before changing the focus.
		synchronized (this) {
			Widget[] widgets = this.widgets;
			for (int i = widgets.length; --i >= 0;) {
				Widget widget = widgets[i];
				widget.setParent(null);
			}
			this.widgets = new Widget[0];
		}
		if (panel != null && panel.getFocus() != this && focusOwner) {
			// a child owns the panel focus
			panel.setFocus(null);
		}
		// the panel needs to be laid out
		invalidate();
	}

	/**
	 * {@inheritDoc}
	 * <p>
	 * Notifies its children widgets that they are shown.
	 */
	@Override
	public void showNotify() {
		super.showNotify();
		for (Widget widget : this.widgets) {
			widget.showNotify();
		}
	}

	/**
	 * {@inheritDoc}
	 * <p>
	 * Notifies its children widgets that they are hidden.
	 */
	@Override
	public void hideNotify() {
		super.hideNotify();
		for (Widget widget : this.widgets) {
			widget.hideNotify();
		}
	}

	/**
	 * Returns the child widget that is at the specified location.
	 * <p>
	 * If {@link Widget#contains(int, int)} is <code>false</code> for this composite, <code>null</code> is returned.
	 * Otherwise, if there is a child for which {@link Widget#contains(int, int)} returns <code>true</code> then the
	 * result of invoking {@link Widget#getWidgetAt(int, int)} on that widget is returned. Otherwise this composite is
	 * returned.
	 * <p>
	 * The location is relative to the location of this composite's parent.
	 *
	 * @param x
	 *            x coordinate
	 * @param y
	 *            y coordinate
	 * @return the widget at the location, <code>null</code> if no widget is found in this composite hierarchy.
	 */
	@Override
	@Nullable
	public Widget getWidgetAt(int x, int y) {
		// equivalent to super.getWidgetAt(x, y) == null
		if (!contains(x, y)) {
			return null;
		}

		int relX = x - this.x;
		int relY = y - this.y;
		// browse children recursively
		Widget[] widgets = this.widgets;
		for (int i = widgets.length; --i >= 0;) {
			Widget at = widgets[i].getWidgetAt(relX, relY);
			if (at != null) {
				return at;
			}
		}
		return this;
	}

	/**
	 * Gets an iterator over the composite children widgets in proper sequence.
	 * <p>
	 * The {@link Iterator#remove()} is not implemented (throws a {@link UnsupportedOperationException}).
	 *
	 * @return an iterator over the composite children widgets in proper sequence.
	 * @since 2.0
	 */
	@NonNull
	public Iterator<Widget> iterator() {
		return new Iterator<Widget>() {
			int cursor = 0;

			@Override
			public boolean hasNext() {
				return this.cursor != Composite.this.widgets.length;
			}

			@Override
			public Widget next() {
				int i = this.cursor;
				Widget[] widgets = Composite.this.widgets;
				if (i < widgets.length) {
					Widget next = widgets[i];
					this.cursor = i + 1;
					return next;
				} else {
					throw new NoSuchElementException();
				}
			}

			@Override
			public void remove() {
				throw new UnsupportedOperationException();
			}
		};
	}

	/**
	 * Gets the widget at the specified position in this composite.
	 *
	 * @param index
	 *            the index of the element to return.
	 * @return the widget at the specified position in this composite.
	 * @throws IndexOutOfBoundsException
	 *             if the index is out of range ((<code>index &lt; 0 || index &gt;= getWidgetsCount()</code>)
	 */
	@NonNull
	public Widget getWidget(int index) throws IndexOutOfBoundsException {
		return this.widgets[index];
	}

	/**
	 * Gets the list of children in this composite.
	 *
	 * @return the list of children.
	 */
	@NonNull
	public Widget[] getWidgets() {
		return (Widget[]) ArrayTools.copy(this.widgets, Widget[].class);
	}

	/**
	 * Gets the number of children in this composite.
	 *
	 * @return the number of children.
	 */
	public int getWidgetsCount() {
		return this.widgets.length;
	}

	@Override
	boolean isFocusOwner() {
		try {
			Widget focus = this.panel.getFocus();
			// search recursively in the hierarchy of the focused widget
			while (focus != null) {
				if (focus == this) {
					return true;
				}
				focus = focus.getParent();
			}
		} catch (NullPointerException e) {
			// Not on a panel.
		}
		return super.isFocusOwner();
	}

	/**
	 * Requests that the first child of this composite be set as the focus owner of its panel.
	 * <p>
	 * If the composite does not contain any widgets, nothing is done.<br>
	 * If the composite is not in a panel hierarchy, nothing is done.
	 * <p>
	 * Identical to calling {@link #requestFocusFrom(int, int)} with {@link MWT#RIGHT} as direction and <code>0</code>
	 * as from.
	 */
	@Override
	public void requestFocus() {
		if (getWidgetsCount() != 0) {
			requestFocusFrom(0, MWT.RIGHT);
		}
	}

	/**
	 * Sets a widget in this composite as the focus owner of its panel, if it is enabled, following the direction.
	 * <p>
	 * The given direction must be one of {@link MWT#UP}, {@link MWT#DOWN}, {@link MWT#LEFT}, {@link MWT#RIGHT}.
	 * <p>
	 * If the widget is not in a panel hierarchy, nothing is done.
	 * <p>
	 * Identical to calling {@link #requestFocusFrom(int, int)} with one of {@link MWT#DOWN} or {@link MWT#RIGHT} as
	 * direction and <code>0</code> as from, or one of {@link MWT#LEFT} or {@link MWT#UP} and
	 * <code>(getWidgetsCount() - 1)</code> as from.
	 *
	 * @param direction
	 *            the direction followed by the focus
	 * @return <code>true</code> if the composite take the focus, <code>false</code> otherwise
	 * @throws IllegalArgumentException
	 *             if <code>direction</code> is not a valid direction
	 */
	@Override
	public boolean requestFocus(int direction) throws IllegalArgumentException {
		int widgetsCount = getWidgetsCount();
		if (widgetsCount != 0 && isEnabled()) {
			switch (direction) {
			case MWT.UP:
			case MWT.LEFT:
				return requestFocusFrom(widgetsCount - 1, direction);
			case MWT.DOWN:
			case MWT.RIGHT:
				return requestFocusFrom(0, direction);
			default:
				throw new IllegalArgumentException();
			}
		}
		return false;
	}

	/**
	 * Gives the focus to the first enabled widget that is in the list of this composite's children from the widget at
	 * the specified index following the direction.
	 * <p>
	 * The given direction must be one of {@link MWT#UP}, {@link MWT#DOWN}, {@link MWT#LEFT}, {@link MWT#RIGHT}.
	 *
	 * @param from
	 *            the index to start search
	 * @param direction
	 *            the direction followed by the focus
	 * @return <code>true</code> if a new widget has been given focus, <code>false</code> otherwise
	 * @throws IndexOutOfBoundsException
	 *             if <code>from</code> is not a valid index
	 * @throws IllegalArgumentException
	 *             if <code>direction</code> is not a valid direction
	 * @see #getNext(int, int)
	 */
	public boolean requestFocusFrom(int from, int direction)
			throws IndexOutOfBoundsException, IllegalArgumentException {
		checkFrom(from);
		do {
			if (getWidget(from).requestFocus(direction)) {
				return true;
			}
			from = getNext(from, direction);
		} while (from != MWT.EMPTY);
		return false;
	}

	/**
	 * Gets the next widget in the focus order following the direction.
	 * <p>
	 * If there is no more widget to focus in this direction, it returns {@link MWT#EMPTY}.
	 * <p>
	 * The given direction must be one of {@link MWT#UP}, {@link MWT#DOWN}, {@link MWT#LEFT}, {@link MWT#RIGHT}.
	 *
	 * @param from
	 *            the index of the current widget
	 * @param direction
	 *            the direction to follow
	 * @return the index of the next widget
	 * @throws IndexOutOfBoundsException
	 *             if <code>from</code> is not a valid index
	 * @throws IllegalArgumentException
	 *             if <code>direction</code> is not a valid direction
	 * @see #getFocusIndex()
	 * @since 1.0
	 */
	public int getNext(int from, int direction) throws IndexOutOfBoundsException, IllegalArgumentException {
		checkFrom(from);
		switch (direction) {
		case MWT.UP:
		case MWT.LEFT:
			return getPrevious(from);
		case MWT.DOWN:
		case MWT.RIGHT:
			return getNext(from);
		default:
			// invalid direction
			throw new IllegalArgumentException();
		}
	}

	/**
	 * Gets the widget that is the focus owner or that owns (recursively) the focus owner in this composite.
	 *
	 * @return the widget that own the focus on this composite or <code>null</code>
	 */
	@Nullable
	public Widget getFocus() {
		try {
			return this.widgets[getFocusIndex()];
		} catch (ArrayIndexOutOfBoundsException e) {
			// not in the hierarchy
		}
		return null;
	}

	/**
	 * Gets the index of the widget that is the focus owner or that is (recursively) the focus owner parent in the
	 * composite. Returns {@link MWT#EMPTY} if the focus owner is not in the composite hierarchy.
	 *
	 * @return the widget that own the focus on this composite or {@link MWT#EMPTY}
	 */
	public int getFocusIndex() {
		Widget[] widgets = this.widgets;
		int widgetsLength = widgets.length;
		for (int i = widgetsLength; --i >= 0;) {
			if (widgets[i].hasFocus()) {
				return i;
			}
		}
		// no focus
		return MWT.EMPTY;
	}

	/**
	 * Lays out this composite and all its children.
	 * <p>
	 * The parameters defines the maximum size available for this composite, or {@link MWT#NONE} if there is no
	 * constraint.
	 * <p>
	 * After this call the preferred size will have been established.
	 *
	 * @param widthHint
	 *            the width available for this widget or {@link MWT#NONE}
	 * @param heightHint
	 *            the height available for this widget or {@link MWT#NONE}
	 */
	@Override
	public abstract void validate(int widthHint, int heightHint);

	/**
	 * Called by the system if a child of this composite is the owner of the focus of the active panel (recursively) and
	 * have not consumed the specified event. Composites handles {@link Command#UP}, {@link Command#DOWN},
	 * {@link Command#LEFT}, and {@link Command#RIGHT} commands to manage navigation in its children.
	 *
	 * @param event
	 *            the event to handle
	 * @return <code>true</code> if the composite consume the event, <code>false</code> otherwise
	 */
	@Override
	public boolean handleEvent(int event) {
		return super.handleEvent(event) || handleDirection(event);
	}

	private boolean handleDirection(int event) {
		int direction = RenderableHelper.getDirection(event);
		if (direction != MWT.NONE) {
			// search the focused widget
			int i = getFocusIndex();
			try {
				i = getNext(i, direction);
				return requestFocusFrom(i, direction);
			} catch (ArrayIndexOutOfBoundsException e) {
				// not the focus owner
				// or the current focused is the first or the last
			}
		}
		return false;
	}

	/* NOT IN API */

	@Override
	/* package */void setPanelOnly(@Nullable Panel panel) {
		super.setPanelOnly(panel);

		// update children recursively
		Widget[] contents = this.widgets;
		for (int i = contents.length; --i >= 0;) {
			contents[i].setPanelOnly(panel);
		}
	}

	private void checkFrom(int from) {
		if (from < 0 || from >= getWidgetsCount()) {
			throw new ArrayIndexOutOfBoundsException();
		}
	}

	private int getNext(int from) {
		if (++from >= getWidgetsCount()) {
			return MWT.EMPTY;
		}
		return from;
	}

	private int getPrevious(int from) {
		return --from; // does not need to check: --0 = MWT.EMPTY
	}

	@Override
	/* package */void paint(@NonNull GraphicsContext g, int translateX, int translateY, int x, int y, int width,
			int height) {
		if (!this.visible) {
			return;
		}
		beforePaint(g, translateX, translateY, x, y, width, height);
		// get clip before calling composite rendering (avoid potential side effects)
		width = g.getClipWidth();
		height = g.getClipHeight();
		if (width == 0 || height == 0) {
			return; // nothing to paint, stop recursion
		}
		translateX = g.getTranslateX();
		translateY = g.getTranslateY();
		x = g.getClipX();
		y = g.getClipY();
		paint(g);
		Widget[] widgets = this.widgets;
		int length = widgets.length;
		// paint the views from the first added to the last (sounds "more obvious")
		for (int i = -1; ++i < length;) {
			widgets[i].paint(g, translateX, translateY, x, y, width, height);
		}
	}

}
