/*
 * Java
 *
 * Copyright 2025 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package com.microej.kf.util.monitoring;

import ej.annotation.Nullable;
import ej.basictool.map.PackedMap;
import ej.bon.Timer;
import ej.bon.TimerTask;
import ej.kf.Feature;
import ej.kf.FeatureStateListener;
import ej.kf.Kernel;
import ej.kf.Module;

/**
 * This class represents the implementation for the ResourceMonitoringService.
 */
public class ResourceMonitoringServiceImpl implements ResourceMonitoringService, FeatureStateListener {
	private static final int NO_QUOTA = -1;
	private static final int CALIBRATION_QUOTA = Integer.getInteger("monitoring.check.cpu.calibration", 100_000);
	private static final float ONE_SECOND = 1000.0f;
	private static final float MAX_USAGE = 1.0f;

	private final Timer timer;
	private final long intervalMillis;
	private final boolean runGcBeforeCollecting;
	private final MonitoringData monitoringData;

	@Nullable
	private TimerTask monitoringTask;
	private boolean running;

	/**
	 * Instantiates the resource monitoring service.
	 *
	 * @param timer
	 * 		the timer.
	 * @param intervalMillis
	 * 		the timer interval read from the kernel properties.
	 * @param runGcBeforeCollecting
	 * 		the boolean read from kernel properties to run GC before collecting monitoring data.
	 */
	public ResourceMonitoringServiceImpl(Timer timer, long intervalMillis, boolean runGcBeforeCollecting) {
		this.timer = timer;
		this.intervalMillis = intervalMillis;
		this.runGcBeforeCollecting = runGcBeforeCollecting;

		Kernel.addFeatureStateListener(this);

		this.monitoringData = new MonitoringData(runGcBeforeCollecting);
		this.monitoringTask = null;
	}

	/**
	 * Starts the resource monitoring task by enabling monitoring on each module.
	 */
	public void start() {
		if (running) {
			return; // Monitoring task already running.
		}

		enableModuleMonitoring(Kernel.getInstance());
		for (Feature module : Kernel.getAllLoadedFeatures()) {
			if (Feature.State.STARTED.equals(module.getState())) {
				enableModuleMonitoring(module);
			}
		}

		monitoringTask = new MonitoringTask(monitoringData, runGcBeforeCollecting);
		timer.scheduleAtFixedRate(monitoringTask, 0, intervalMillis);

		running = true;
	}

	/**
	 * Stops resource monitoring by disabling monitoring on each module.
	 */
	public void stop() {
		if (!running) {
			return;
		}

		monitoringTask.cancel();

		disableModuleMonitoring(Kernel.getInstance());
		for (Feature module : Kernel.getAllLoadedFeatures()) {
			if (Feature.State.STARTED.equals(module.getState())) {
				disableModuleMonitoring(module);
			}
		}

		running = false;
	}

	@Override
	public void stateChanged(Feature app, @Nullable Feature.State state) {
		final Feature.State currentState = app.getState();
		if (Feature.State.STARTED.equals(currentState)) {
			enableModuleMonitoring(app);
		} else if (Feature.State.STOPPED.equals(currentState)) {
			disableModuleMonitoring(app);
		}
	}

	@Override
	public PackedMap<String, Float> getUsedCpuPercentPerModule() {
		PackedMap<String, Float> result = new PackedMap<>();

		synchronized (monitoringData) {
			long totalExecutionCounter = 0;
			for (MonitoringData.ExecutionCounter execCounter : monitoringData.getExecutionCounterPerModuleMap()
					.values()) {
				totalExecutionCounter += (execCounter.getCurrent() - execCounter.getLast());
			}

			float totalShare = 0.0f;
			for (Module module : monitoringData.getExecutionCounterPerModuleMap().keySet()) {
				MonitoringData.ExecutionCounter execCounter = monitoringData.getExecutionCounterPerModuleMap()
						.get(module);
				if (totalExecutionCounter > 0) {
					long delta = execCounter.getCurrent() - execCounter.getLast();
					// We might lose precision by converting long to float.
					// Consider that calibration represents an execution counter per second, and the polling interval is
					// in milliseconds.
					float usageShare = (((float) delta / intervalMillis) * ONE_SECOND) / CALIBRATION_QUOTA;
					totalShare += usageShare;
					result.put(module.getName(), usageShare);
				} else {
					// The execution counter was reset, or something went really wrong.
					result.put(module.getName(), 0.0f);
				}
			}

			// Since the calibration is an empirical value, the total shares might exceed 1.0.
			if (totalShare > MAX_USAGE) {
				float factor = MAX_USAGE / totalShare; // NOSONAR (S3518) - False positive, value can't be 0.0f.
				for (String module : result.keySet()) {
					float value = result.get(module);
					result.put(module, value * factor);
				}
			}
		}

		return result;
	}

	@Override
	public PackedMap<String, Long> getUsedMemoryPerModule() {
		PackedMap<String, Long> result = new PackedMap<>();
		synchronized (monitoringData) {
			for (Module module : monitoringData.getRamUsagePerModuleMap().keySet()) {
				result.put(module.getName(), monitoringData.getRamUsagePerModuleMap().get(module));
			}
		}

		return result;
	}

	@Override
	public PackedMap<String, Float> getUsedMemoryPercentPerModule() {
		PackedMap<String, Float> result = new PackedMap<>();
		synchronized (monitoringData) {
			for (Module module : monitoringData.getRamUsagePerModuleMap().keySet()) {
				long value = monitoringData.getRamUsagePerModuleMap().get(module);
				result.put(module.getName(), (float) (value) / monitoringData.getTotalMemory());
			}
		}
		return result;
	}

	@Override
	public long getFreeMemory() {
		return monitoringData.getFreeMemory();
	}

	@Override
	public long getMaxMemory() {
		return monitoringData.getMaxMemory();
	}

	@Override
	public long getTotalMemory() {
		return monitoringData.getTotalMemory();
	}

	/**
	 * Enables monitoring for a given module (kernel or feature). Adds the initial value for the module in the
	 * monitoringData map.
	 *
	 * @param module
	 * 		the module to monitor.
	 */
	private void enableModuleMonitoring(Module module) {
		int previousQuota = module.getExecutionQuota();
		boolean noQuota = previousQuota == NO_QUOTA;
		if (noQuota) {
			// Keep default quota at -1. This disables CPU monitoring for the given module (even the kernel).
			module.setExecutionQuota(NO_QUOTA);
		}
		synchronized (monitoringData) {
			monitoringData.getOriginalQuotaPerModuleMap().put(module, previousQuota);
			monitoringData.getExecutionCounterPerModuleMap().put(module, new MonitoringData.ExecutionCounter(0, 0));
			monitoringData.getRamUsagePerModuleMap().put(module, 0L);
		}
	}

	/**
	 * Disables monitoring for a given module (kernel or feature). Removes related data from the monitoringData map.
	 *
	 * @param module
	 * 		the module to stop monitoring.
	 */
	private void disableModuleMonitoring(Module module) {
		synchronized (monitoringData) {
			monitoringData.getExecutionCounterPerModuleMap().remove(module);
			monitoringData.getRamUsagePerModuleMap().remove(module);
			Integer originalQuota = monitoringData.getOriginalQuotaPerModuleMap().remove(module);
			if (originalQuota != null) {
				module.setExecutionQuota(originalQuota);
			}
		}
	}

	/**
	 * This class represents the monitoring task ran at regular interval.
	 */
	public static final class MonitoringTask extends TimerTask {

		private final MonitoringData monitoringData;
		private final boolean runGcBeforeCollecting;

		/**
		 * Instantiates the monitoring task.
		 *
		 * @param monitoringData
		 * 		the data structure to regroup all the monitoring data.
		 * @param runGcBeforeCollecting
		 * 		the boolean to call the GC before collecting new monitoring data.
		 */
		public MonitoringTask(final MonitoringData monitoringData, boolean runGcBeforeCollecting) {
			this.monitoringData = monitoringData;
			this.runGcBeforeCollecting = runGcBeforeCollecting;
		}

		@Override
		public void run() {
			if (runGcBeforeCollecting) {
				Runtime.getRuntime().gc(); // NOSONAR - Call used to increase heap monitoring accuracy.
			}

			for (Feature app : Kernel.getAllLoadedFeatures()) {
				if (app.getState() == Feature.State.STARTED) {
					computeModuleCpuUsage(app);
					computeModuleMemoryUsage(app);
				}
			}

			computeModuleCpuUsage(Kernel.getInstance());
			computeModuleMemoryUsage(Kernel.getInstance());
		}

		/**
		 * Retrieves and updates CPU usage for the given module.
		 *
		 * @param module
		 * 		the module to monitor.
		 */
		private void computeModuleCpuUsage(Module module) {
			synchronized (monitoringData) {
				MonitoringData.ExecutionCounter exec = monitoringData.getExecutionCounterPerModuleMap().get(module);
				if (exec != null) {
					long newExecutionCounter = module.getExecutionCounter();
					// If new execution counter is less than last, the execution counter was reset.
					exec.setLast(newExecutionCounter < exec.getCurrent() ? 0 : exec.getCurrent());
					exec.setCurrent(newExecutionCounter);
				}
			}
		}

		/**
		 * Retrieves and updates memory usage for the given module.
		 *
		 * @param module
		 * 		the module to monitor.
		 */
		private void computeModuleMemoryUsage(Module module) {
			synchronized (monitoringData) {
				monitoringData.getRamUsagePerModuleMap().put(module, module.getAllocatedMemory());
			}
		}
	}

}
