/*
 * Java
 *
 * Copyright 2025 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package com.microej.kf.util.control.fs;

import java.io.IOException;
import java.io.RandomAccessFile;

import com.microej.kf.util.module.SandboxedModule;
import com.microej.kf.util.module.SandboxedModuleHelper;

import ej.service.ServiceFactory;

/**
 * Action performed when writing to a RandomAccessFile.
 */
public class WriteRandomAccessFile {

	private final long nbWriteBytes;

	/**
	 * Creates a write file action.
	 *
	 * @param file
	 * 		the file to write.
	 * @param nbWriteBytes
	 * 		the number of bytes to write.
	 * @throws IOException
	 * 		if I/O errors occur.
	 */
	public WriteRandomAccessFile(RandomAccessFile file, int nbWriteBytes) throws IOException {
		long length = file.length();
		long position = file.getFilePointer();
		long overwriteLength = length - position;
		long remainingWriteLength = nbWriteBytes - overwriteLength;

		this.nbWriteBytes = (remainingWriteLength < 0) ? 0 : remainingWriteLength;
	}

	/**
	 * Gets the number of bytes to write.
	 *
	 * @return the number of bytes to write.
	 */
	public long getNbBytes() {
		return this.nbWriteBytes;
	}

	/**
	 * Called when the execution of the action is about to start.
	 *
	 * This hook is used to check if the action is allowed to be performed.
	 *
	 * @throws SecurityException
	 * 		if the action is not allowed to be performed.
	 */
	public void onStart() throws SecurityException {
		SandboxedModuleHelper cmh = ServiceFactory.getRequiredService(SandboxedModuleHelper.class);
		SandboxedModule currentModule = cmh.getCurrentModule();
		currentModule.getFileSystemResourceController().onStart(currentModule, this);
	}

	/**
	 * Called when the execution of the action is finished.
	 *
	 * This hook is used to update the current resources state of the module that performs the action.
	 *
	 * @param endWithSuccess
	 * 		true if the execution of the action was ended with success; false otherwise.
	 */
	public void onEnd(boolean endWithSuccess) {
		SandboxedModuleHelper cmh = ServiceFactory.getRequiredService(SandboxedModuleHelper.class);
		SandboxedModule currentModule = cmh.getCurrentModule();
		currentModule.getFileSystemResourceController().onEnd(currentModule, this, endWithSuccess);
	}
}
