/*
 * Java
 *
 * Copyright 2015-2025 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package com.microej.kf.util;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;

import ej.kf.Feature;
import ej.kf.Kernel;

/**
 * Utility functions on top of <code>KF</code> library.
 */
public class KernelSupport {

	/**
	 * KF-1.3 spec: <code>Feature.stop()</code> may result to <code>STOPPED</code> state when there are remaining
	 * references from Kernel to the Feature being killed. Wadapps API does not provide this intermediate step (the
	 * <code>STOPPED</code> state is not a Wadapps state). This function blocks until all Feature references are removed
	 * by the Kernel (i.e. wait until pumps have finished their current job, ...) or the timeout occurs.
	 *
	 * @param f
	 *            the feature to stop.
	 * @param timeout
	 *            the maximum time to wait in milliseconds.
	 * @return true if Feature has been successfully stopped, false if timeout occurred (the Feature remains in
	 *         <code>STOPPED</code> state)
	 */
	public static boolean stopFeature(Feature f, long timeout) {
		int delay = 250;
		long start = System.currentTimeMillis();
		while (true) {
			try {
				f.stop();
			} catch (IllegalStateException e) {
				// an other thread may have stopped or uninstalled the Feature.
			}
			if (f.getState() != Feature.State.STOPPED) {
				// may be in INSTALLED state
				// may also be in STARTED or UNINSTALLED if done by an other thread
				// => terminate
				return true;
			}
			if (System.currentTimeMillis() - start >= timeout) {
				return false;
			}

			try {
				Thread.sleep(delay);
			} catch (InterruptedException ignored) {
				// continue
			}
		}
	}

	/**
	 * This function infinitely blocks until all Feature references are removed by the Kernel.
	 */
	public static void stopFeature(Feature f) {
		stopFeature(f, Long.MAX_VALUE);
	}

	static List<Class<Collection<?>>> knownCollectionClasses;

	@SuppressWarnings({ "unchecked" })
	public static <T> T clone(T source, Feature cloneOwner) {
		// let's try first if bind can succeed
		try {
			return Kernel.bind(source, (Class<T>) source.getClass(), cloneOwner);
		} catch (IllegalAccessError e) {
			// if not it might be because we're trying to bind using the runtime time
			// e.g. ArrayList.class, whereas the converter is registered on List.class
			synchronized (KernelSupport.class) {
				// is the list of known such cases already defined ?
				if (knownCollectionClasses == null) {
					// if not let's populate it
					knownCollectionClasses = new ArrayList<>();
					knownCollectionClasses.add((Class<Collection<?>>) (Object) Map.class);
					knownCollectionClasses.add((Class<Collection<?>>) (Object) List.class);
				}
			}
			// now try the bind again, this time with one of these types, if applicable
			for (Class<Collection<?>> c : knownCollectionClasses) {
				if (c.isInstance(source)) {
					return Kernel.bind(source, (Class<T>) c, cloneOwner);
				}
			}
			// no compatible converter found, let's re throw the error
			throw e;
		}
	}

}
