/*
 * Java
 *
 * Copyright 2019 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.net.util.connectivity;

import android.net.ConnectivityManager.NetworkCallback;
import android.net.Network;
import android.net.NetworkCapabilities;
import ej.annotation.Nullable;

/**
 * A simple network call back adapter. It can be extended or used with a listener.
 */
public class SimpleNetworkCallbackAdapter extends NetworkCallback implements SimpleNetworkCallback {

	private boolean hasInternet;

	private boolean isConnected;

	private SimpleNetworkCallback listener;

	/**
	 * Instantiates a {@link SimpleNetworkCallback}.
	 */
	public SimpleNetworkCallbackAdapter() {
		// Nothing to do.
	}

	/**
	 * Instantiates a {@link SimpleNetworkCallback}.
	 *
	 * @param listener
	 *            the listener to call onstate change.
	 */
	public SimpleNetworkCallbackAdapter(@Nullable SimpleNetworkCallback listener) {
		this.listener = listener;
	}

	/**
	 * Gets the listener to call back on state change.
	 *
	 * @return the listener, <code>null</code> if none.
	 */
	public SimpleNetworkCallback getListener() {
		return this.listener;
	}

	/**
	 * Sets the listener to call back on state change.
	 *
	 * @param listener
	 *            the listener to set.
	 */
	public void setListener(@Nullable SimpleNetworkCallback listener) {
		this.listener = listener;
	}

	@Override
	public void onAvailable(Network network) {
		this.isConnected = true;
		onConnectivity(true);
	}

	@Override
	public void onLost(Network network) {
		this.isConnected = false;
		onConnectivity(false);
	}

	@Override
	public void onCapabilitiesChanged(Network network, NetworkCapabilities networkCapabilities) {
		this.hasInternet = networkCapabilities.hasCapability(NetworkCapabilities.NET_CAPABILITY_INTERNET);
		onInternet(this.hasInternet);
	}

	/**
	 * Called when the Internet state change.
	 *
	 * @param hasInternet
	 *            <code>true</code> if Internet access is available.
	 * @see #onCapabilitiesChanged(Network, NetworkCapabilities)
	 */
	@Override
	public void onInternet(boolean hasInternet) {
		SimpleNetworkCallback listener = this.listener;
		if (listener != null) {
			listener.onInternet(hasInternet);
		}
	}

	/**
	 * Called when the connectivity state changes.
	 *
	 * @param isConnected
	 *            <code>true</code> if a network is available.
	 */
	@Override
	public void onConnectivity(boolean isConnected) {
		SimpleNetworkCallback listener = this.listener;
		if (listener != null) {
			listener.onConnectivity(isConnected);
		}
	}

	/**
	 * Checks whether the call back had been called for Internet.
	 *
	 * @return <code>true</code> if the last call of {@link #onInternet(boolean)} had Internet.
	 */
	public boolean hasInternet() {
		return this.hasInternet;
	}

	/**
	 * Checks whether the call back had been called for connection.
	 *
	 * @return <code>true</code> if the last call of {@link #onConnectivity(boolean)} was connected.
	 */
	public boolean isConnected() {
		return this.isConnected;
	}
}
