/*
 * Java
 *
 * Copyright 2019 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.bluetooth.util.server;

import ej.basictool.ArrayTools;
import ej.bluetooth.BluetoothCharacteristic;
import ej.bluetooth.BluetoothDescriptor;
import ej.bluetooth.BluetoothDevice;
import ej.bluetooth.BluetoothService;
import ej.bluetooth.BluetoothUuid;
import ej.bluetooth.callbacks.ServerCallbacks;

/**
 * A {@link BluetoothService} that can be notified and can forward read and write request to its characteritics.
 */
public class BasicBluetoothServerService extends BluetoothService implements ServerCallbacks {

	private BasicServerCharacteristic[] characteristics;

	/**
	 * instantiates a {@link BasicBluetoothServerService}.
	 *
	 * @param uuid
	 *            the uuid of the service.
	 * @param type
	 *            the type of service.
	 */
	public BasicBluetoothServerService(BluetoothUuid uuid, Type type) {
		super(uuid, type);
		setServerCallbacks(this);
		this.characteristics = new BasicServerCharacteristic[0];
	}

	/**
	 * Add a characteristic that can received write and read request.
	 *
	 * @param characteristic
	 *            the characteristic to add.
	 */
	public void addCharacteristic(BasicServerCharacteristic characteristic) {
		super.addCharacteristic(characteristic);
		this.characteristics = ArrayTools.add(this.characteristics, characteristic);
	}

	@Override
	public void onReadRequest(BluetoothCharacteristic characteristic, BluetoothDevice device) {
		for (BasicServerCharacteristic bluetoothCharacteristic : this.characteristics) {
			if (characteristic.equals(bluetoothCharacteristic)) {
				bluetoothCharacteristic.onReadRequest(device);
			}
		}
	}

	@Override
	public void onWriteRequest(BluetoothCharacteristic characteristic, BluetoothDevice device, byte[] value) {
		for (BasicServerCharacteristic bluetoothCharacteristic : this.characteristics) {
			if (characteristic.equals(bluetoothCharacteristic)) {
				bluetoothCharacteristic.onWriteRequest(device, value);
			}
		}
	}

	@Override
	public void onNotificationSent(BluetoothCharacteristic characteristic, BluetoothDevice device, boolean success) {
		// nothing to do.

	}

	@Override
	public void onReadRequest(BluetoothDescriptor descriptor, BluetoothDevice device) {
		for (BasicServerCharacteristic bluetoothCharacteristic : this.characteristics) {
			for (BasicServerDescriptor bluetoothDescriptor : bluetoothCharacteristic.getServerDescriptors()) {
				if (descriptor.equals(bluetoothDescriptor)) {
					bluetoothDescriptor.onReadRequest(device);
				}
			}
		}
	}

	@Override
	public void onWriteRequest(BluetoothDescriptor descriptor, BluetoothDevice device, byte[] value) {
		for (BasicServerCharacteristic bluetoothCharacteristic : this.characteristics) {
			for (BasicServerDescriptor bluetoothDescriptor : bluetoothCharacteristic.getServerDescriptors()) {
				if (descriptor.equals(bluetoothDescriptor)) {
					bluetoothDescriptor.onWriteRequest(device, value);
				}
			}
		}
	}
}
