/*
 * Copyright 2022 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package com.google.protobuf;

import java.io.IOException;

import com.google.protobuf.GeneratedMessageLite.ExtendableBuilder;
import com.google.protobuf.GeneratedMessageLite.ExtendableMessage;
import com.google.protobuf.GeneratedMessageLite.ExtensionDescriptor;
import com.google.protobuf.Internal.EnumLite;
import com.google.protobuf.Internal.EnumLiteMap;
import com.google.protobuf.WireFormat.FieldType;

import ej.annotation.Nullable;

/**
 * Provides a utility method to initialize Protobuf library when it is embedded in a Kernel and its APIs are exposed to
 * Features.
 */
public class ProtobufKernel {

	/* package */ interface MethodCalledListener {

		void onMethodCalled(StackTraceElement element);
	}

	/* package */ static final String DEBUG_SWITCH_TABLES_CONSTANT = "com.google.protobuf.DEBUG_SWITCH_TABLES"; //$NON-NLS-1$

	private static @Nullable MethodCalledListener methodCalledListener;

	private ProtobufKernel() {
		// private constructor
	}

	/**
	 * When the Protobuf library is exposed as Kernel API, this method should be called before a Feature calls one of
	 * the Protobuf APIs.
	 */
	public static void initialize() {
		// Initializes the switch tables of switch statements on enums.

		// call FieldSet.computeElementSizeNoTag()
		ExtensionDescriptor descriptor = new ExtensionDescriptor(createEnumLiteMap(), 0, FieldType.BOOL, false, false);
		FieldSet.computeFieldSize(descriptor, Boolean.TRUE);

		// call FieldSet.writeElementNoTag()
		try {
			CodedOutputStream outputStream = CodedOutputStream.newInstance(new byte[2]);
			FieldSet.writeField(descriptor, Boolean.TRUE, outputStream);
		} catch (IOException e) {
			// ignore exception
		}

		// call GeneratedMessageLite$ExtendableMessage.parseUnknownField(), WireFormat.readPrimitiveField() and
		// FieldSet.verifyType()
		try {
			CodedInputStream inputStream = CodedInputStream.newInstance(new byte[4]);
			MyExtendableMessage message = new MyExtendableMessage();
			ExtensionRegistryLite registry = ExtensionRegistryLite.newInstance();
			registry.add(GeneratedMessageLite.newSingularGeneratedExtension(message, Boolean.TRUE, message,
					createEnumLiteMap(), 0, FieldType.BOOL, Boolean.class));
			message.parseUnknownField(message, inputStream, registry, WireFormat.WIRETYPE_VARINT);
		} catch (IOException e) {
			// ignore exception
		}

		// call MapEntryLite.parseField() and WireFormat.readPrimitiveField()
		try {
			CodedInputStream inputStream = CodedInputStream.newInstance(new byte[1]);
			MapEntryLite.parseField(inputStream, ExtensionRegistryLite.newInstance(), FieldType.BOOL, Boolean.TRUE);
		} catch (IOException e) {
			// ignore exception
		}
	}

	/* package */ static void setMethodCalledListener(@Nullable MethodCalledListener listener) {
		ProtobufKernel.methodCalledListener = listener;
	}

	/* package */ static void onMethodCalled() {
		MethodCalledListener listener = ProtobufKernel.methodCalledListener;
		if (listener != null) {
			StackTraceElement[] stackTrace = Thread.currentThread().getStackTrace();
			String className = ProtobufKernel.class.getName();
			for (int i = 0; i < stackTrace.length; i++) {
				if (stackTrace[i].getClassName().equals(className)) {
					StackTraceElement element = stackTrace[i + 1];
					assert (element != null);
					listener.onMethodCalled(element);
					break;
				}
			}
		}
	}

	private static class MyExtendableBuilder extends ExtendableBuilder<MyExtendableMessage, MyExtendableBuilder> {

		protected MyExtendableBuilder(MyExtendableMessage defaultInstance) {
			super(defaultInstance);
		}
	}

	private static class MyExtendableMessage extends ExtendableMessage<MyExtendableMessage, MyExtendableBuilder> {

		@Override
		public void writeTo(CodedOutputStream output) throws IOException {
			// do nothing
		}

		@Override
		public int getSerializedSize() {
			return 0;
		}

		@Override
		protected @Nullable Object dynamicMethod(MethodToInvoke method, @Nullable Object arg0, @Nullable Object arg1) {
			return null;
		}
	}

	private static EnumLiteMap<EnumLite> createEnumLiteMap() {
		return new EnumLiteMap<EnumLite>() {
			@Override
			public EnumLite findValueByNumber(int number) {
				return createEnumLite();
			}
		};
	}

	private static EnumLite createEnumLite() {
		return new EnumLite() {
			@Override
			public int getNumber() {
				return 0;
			}
		};
	}
}
