/*
 * Java
 *
 * Copyright 2014-2019 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.websocket;

/**
 * WebSocketURI is a container for a WebSocket URI as described in RFC 6455. See section 3 (and section 4.1).
 * <p>
 * Limitations of this implementation:
 * <ul>
 * <li>Queries with ? are not checked.
 * <li>Fragments are not checked.
 * <li>'#' character is not managed.
 * </ul>
 *
 *
 *
 */
public class WebSocketURI {

	// Section 3 says that resource name is ""/" if the path component is empty"
	private static final String DEFAULT_RESOURCE_NAME = "/";

	// Section 3 says that port is optional and defines default ports for ws:// et wss://
	/**
	 * The default port for a ws:// URI as defined in RFC6455 (see section 3). If port is not specified during URI
	 * creation, then this port will be used.
	 */
	public static final int DEFAULT_NOT_SECURE_PORT = 80;

	/**
	 * The default port for a wss:// URI. If port is not specified during URI creation, then this port will be used.
	 */
	public static final int DEFAULT_SECURE_PORT = 443;

	// Section 4.1 (page 15)
	private final String host;

	// Section 3: port is optional
	private final int port;

	// Sections 3 and 4.1
	private final String resourceName;

	// Section 3: secure flag is set when using wss
	private final boolean secureFlag;

	/**
	 * Create a new websocket URI.
	 *
	 * @param host
	 *            hostname or IP address of the remote host
	 * @param port
	 *            port of the remote host
	 * @param resourceName
	 *            identifier for the service provided by the server
	 * @param isSecure
	 *            flag indicating if the websocket is secure (wss) or not
	 * @throws IllegalArgumentException
	 *             if host is null or empty; if port is 0 (zero); if
	 *             resourceName is null
	 */
	public WebSocketURI(String host, int port, String resourceName, boolean isSecure) throws IllegalArgumentException {
		// Refuse invalid parameters
		if (host == null || host.isEmpty() || port == 0 || resourceName == null) {
			throw new IllegalArgumentException();
		}

		// Save parameters
		this.host = host;
		this.port = port;
		// Resource name must start with a "/" so that the HTTP request will be valid
		if (resourceName.startsWith(DEFAULT_RESOURCE_NAME)) {
			this.resourceName = resourceName;
		} else {
			this.resourceName = DEFAULT_RESOURCE_NAME + resourceName;
		}

		// Fields not given in constructor parameters
		this.secureFlag = isSecure;
	}

	/**
	 * Create a new (not secure) websocket URI.
	 *
	 * @param host
	 *            hostname or IP address of the remote host
	 * @param port
	 *            port of the remote host
	 * @param resourceName
	 *            identifier for the service provided by the server
	 * @throws IllegalArgumentException
	 *             if host is null or empty; if port is 0 (zero); if
	 *             resourceName is null
	 */
	public WebSocketURI(String host, int port, String resourceName) throws IllegalArgumentException {
		this(host, port, resourceName, false);
	}

	/**
	 * Create a new (not secure) websocket URI. Default port ({@link #DEFAULT_NOT_SECURE_PORT}) is used.
	 *
	 * @param host
	 *            hostname or IP address of the remote host
	 * @param resourceName
	 *            identifier for the service provided by the server
	 * @throws IllegalArgumentException
	 *             if resourceName is null
	 */
	public WebSocketURI(String host, String resourceName) throws IllegalArgumentException {
		this(host, DEFAULT_NOT_SECURE_PORT, resourceName);
	}

	/**
	 * Create a new (not secure) websocket URI. Resource name is assumed to be "/".
	 *
	 * @param host
	 *            hostname or IP address of the remote host
	 * @param port
	 *            port of the remote host
	 * @throws IllegalArgumentException
	 *             if resourceName is null
	 */
	public WebSocketURI(String host, int port) throws IllegalArgumentException {
		this(host, port, DEFAULT_RESOURCE_NAME);
	}

	/**
	 * Create a new (not secure) websocket URI. Resource name is assumed to be "/". Default port ({@link #DEFAULT_NOT_SECURE_PORT}) is used.
	 *
	 * @param host
	 *            hostname or IP address of the remote host
	 * @throws IllegalArgumentException
	 *             if resourceName is null
	 */
	public WebSocketURI(String host) throws IllegalArgumentException {
		this(host, DEFAULT_NOT_SECURE_PORT, DEFAULT_RESOURCE_NAME);
	}

	/**
	 * Get the host (hostname or IP address).
	 *
	 * @return hostname of the peer
	 */
	public String getHost() {
		return host;
	}

	/**
	 * Get the port of the host.
	 *
	 * @return port of the remote peer
	 */
	public int getPort() {
		return port;
	}

	/**
	 * Get the resource name, that is the name of the remote websocket endpoint.
	 *
	 * @return the request-URI of the remote endpoint
	 */
	public String getResourceName() {
		return resourceName;
	}

	/**
	 * Tell if the websocket is secured or not.
	 *
	 * @return true is this URI is intended to be create a secured connection (wss://); false otherwise (ws://)
	 */
	public boolean isSecure() {
		return secureFlag;
	}

	@Override
	public String toString() {
		String prefix = secureFlag ? "wss://" : "ws://";
		return prefix + host + ":" + port + resourceName;
	}

}
