/*
 * Copyright 2023-2024 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package ej.fp.widget.display;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import ej.fp.FrontPanel;
import ej.fp.Image;
import ej.fp.Widget;
import ej.fp.widget.display.brs.BufferRefreshStrategy;
import ej.fp.widget.display.buffer.DisplayBufferPolicy;
import ej.microui.display.Rectangle;

/**
 * The display buffer manager is responsible of the drawings and flushing on the right buffer(s).
 */
public class DisplayBufferManager implements Image {

	private static final int WHITE_OPAQUE = 0xffffffff;
	private static final int BLACK_OPAQUE = 0xff000000;
	private static final int FULLY_TRANSPARENT = 0x00000000;
	private static final int MAX_COLOR_COMPONENT_VALUE = 0xff;
	private static final int ALPHA_SHIFT = 24;
	private static final int RED_SHIFT = 16;
	private static final int GREEN_SHIFT = 8;
	private static final int BLUE_SHIFT = 0;
	private static final int ALPHA_MASK = MAX_COLOR_COMPONENT_VALUE << ALPHA_SHIFT;
	private static final int RED_MASK = MAX_COLOR_COMPONENT_VALUE << RED_SHIFT;
	private static final int GREEN_MASK = MAX_COLOR_COMPONENT_VALUE << GREEN_SHIFT;
	private static final int BLUE_MASK = MAX_COLOR_COMPONENT_VALUE << BLUE_SHIFT;
	private static final int LIGHT_COLOR_LIMIT = 0x7f;
	private static final float BLUE_LIGHT_FACTOR = 0.114f;
	private static final float GREEN_LIGHT_FACTOR = 0.587f;
	private static final float RED_LIGHT_FACTOR = 0.299f;

	private static final String DEBUG_DRAWN_COLOR = "ej.fp.brs.drawnColor"; //$NON-NLS-1$
	private static final String DEBUG_RESTORE_COLOR = "ej.fp.brs.restoredColor"; //$NON-NLS-1$
	private static final String DEBUG_DIRTY_COLOR = "ej.fp.brs.dirtyColor"; //$NON-NLS-1$
	private static final int DEBUG_COLOR_WIDTH = 8;
	private static final int DEBUG_COLOR_HEIGHT = 14;

	private final Widget displayWidget;
	private final DisplayBufferPolicy bufferPolicy;
	private final BufferRefreshStrategy refreshStrategy;
	private boolean hasFlushed;

	private final Object monitor;

	// Third-party thread that simulates the embedded display.
	private final FlushThread flushThread;

	private int flushTimeMs;
	private int refreshRateTimeMs;

	// Debug stuff
	private final Image debugBuffer;

	private final boolean debugDrawn;
	private final int drawnColor;
	private final int drawnForeground;
	private final List<Rectangle> drawnRectangles;

	private final boolean debugRestored;
	private final int restoredColor;
	private final int restoredForeground;
	private final List<Rectangle> restoredRectangles;

	private final boolean debugDirty;
	private final int dirtyColor;
	private final Rectangle dirtyRectangle;
	private Image dirtyRectangleBuffer;

	/**
	 * Creates a buffer manager.
	 */
	public DisplayBufferManager(DisplayBufferPolicy bufferPolicy, BufferRefreshStrategy refreshStrategy,
			Widget displayWidget, int displayWidth, int displayHeight, int initialColor) {
		this.bufferPolicy = bufferPolicy;
		this.refreshStrategy = refreshStrategy;
		this.displayWidget = displayWidget;

		bufferPolicy.setDisplayProperties(displayWidget, displayWidth, displayHeight, initialColor);
		refreshStrategy.setBufferCount(bufferPolicy.getBufferCount());
		refreshStrategy.setDisplaySize(displayWidth, displayHeight);

		this.monitor = new Object();

		this.flushThread = new FlushThread();
		this.flushThread.setPriority(Thread.MIN_PRIORITY);
		this.flushThread.start();

		this.debugBuffer = FrontPanel.getFrontPanel().newImage(displayWidth, displayHeight, initialColor, false);

		this.dirtyColor = getColorProperty(DEBUG_DIRTY_COLOR);
		this.debugDirty = !isFullyTransparentColor(this.dirtyColor);
		this.dirtyRectangle = new Rectangle(Integer.MAX_VALUE, Integer.MAX_VALUE, Integer.MIN_VALUE, Integer.MIN_VALUE);

		this.drawnColor = getColorProperty(DEBUG_DRAWN_COLOR);
		this.debugDrawn = isSetColorProperty(DEBUG_DRAWN_COLOR);
		this.drawnForeground = getForegroundColor(this.drawnColor);
		this.drawnRectangles = new ArrayList<>(0);

		this.restoredColor = getColorProperty(DEBUG_RESTORE_COLOR);
		this.debugRestored = isSetColorProperty(DEBUG_RESTORE_COLOR);
		this.restoredForeground = getForegroundColor(this.restoredColor);
		this.restoredRectangles = new ArrayList<>(0);
	}

	/**
	 * Gets whether or not there are several buffers for the display.
	 * <p>
	 * In other words, if the drawings are done in a buffer different than the one used by the display or not.
	 *
	 * @return <code>true</code> if there is more than one buffer, <code>false</code> otherwise.
	 */
	public boolean isDoubleBuffered() {
		return this.bufferPolicy.isDoubleBuffered();
	}

	/**
	 * Gets the monitor.
	 *
	 * @return the monitor.
	 */
	public Object getMonitor() {
		return this.monitor;
	}

	/**
	 * Gets the display image.
	 *
	 * @return the display image.
	 */
	public Image getDisplayImage() {
		Image displayImage = this.bufferPolicy.getFrontBuffer();

		if (this.debugDrawn || this.debugRestored) {
			// recopy the front buffer over the debug buffer to not modify the front buffer
			this.debugBuffer.drawImage(displayImage);
			displayImage = this.debugBuffer;

			addRestoredLayer(displayImage);
			addDrawLayer(displayImage);
		}

		return displayImage;
	}

	private void addDrawLayer(Image displayImage) {
		if (this.debugDrawn) {
			// draw a rectangle around the new regions and a cross to visualize them
			ArrayList<Rectangle> drawnRectangles = new ArrayList<>(this.drawnRectangles);
			int drawnRectanglesLength = drawnRectangles.size();
			for (int i = 0; i < drawnRectanglesLength; i++) {
				Rectangle rectangle = drawnRectangles.get(i);
				displayImage.drawRectangle(rectangle.getX1(), rectangle.getY1(), rectangle.getX2() - rectangle.getX1(),
						rectangle.getY2() - rectangle.getY1(), this.drawnColor);
				displayImage.drawLine(rectangle.getX1() + (rectangle.getX2() - rectangle.getX1()) / 2,
						rectangle.getY1(), rectangle.getX1() + (rectangle.getX2() - rectangle.getX1()) / 2,
						rectangle.getY2(), this.drawnColor);
				displayImage.drawLine(rectangle.getX1(),
						rectangle.getY1() + (rectangle.getY2() - rectangle.getY1()) / 2, rectangle.getX2(),
						rectangle.getY1() + (rectangle.getY2() - rectangle.getY1()) / 2, this.drawnColor);
				displayImage.drawString(Integer.toString(i),
						rectangle.getX1() + (rectangle.getX2() - rectangle.getX1()) / 2,
						rectangle.getY1() + (rectangle.getY2() - rectangle.getY1()) / 2 - DEBUG_COLOR_HEIGHT / 2,
						this.drawnForeground, this.drawnColor);
			}
		}
	}

	private void addRestoredLayer(Image displayImage) {
		if (this.debugRestored) {
			// draw a rectangle around the restored regions and a cross to visualize them
			ArrayList<Rectangle> restoredRectangles = new ArrayList<>(this.restoredRectangles);
			int restoredRectanglesLength = restoredRectangles.size();
			for (int i = 0; i < restoredRectanglesLength; i++) {
				Rectangle rectangle = restoredRectangles.get(i);
				displayImage.drawRectangle(rectangle.getX1(), rectangle.getY1(), rectangle.getX2() - rectangle.getX1(),
						rectangle.getY2() - rectangle.getY1(), this.restoredColor);
				displayImage.drawLine(rectangle.getX1(), rectangle.getY1(), rectangle.getX2(), rectangle.getY2(),
						this.restoredColor);
				displayImage.drawLine(rectangle.getX2(), rectangle.getY1(), rectangle.getX1(), rectangle.getY2(),
						this.restoredColor);
				displayImage.drawString(Integer.toString(i),
						rectangle.getX1() + (rectangle.getX2() - rectangle.getX1()) / 2 - DEBUG_COLOR_WIDTH,
						rectangle.getY1() + (rectangle.getY2() - rectangle.getY1()) / 2 - DEBUG_COLOR_HEIGHT / 2,
						this.restoredForeground, this.restoredColor);
			}
		}
	}

	/**
	 * Gets the current drawing buffer.
	 *
	 * @return the current drawing buffer.
	 */
	public Image getCurrentBackBuffer() {
		return this.bufferPolicy.getBackBuffer();
	}

	/**
	 * Flushes a part of the display limited by the specified bounds when the display is double buffered.
	 */
	public void flush() {
		this.flushThread.wakeup();
	}

	/**
	 * Simulates the waiting of end of an asynchronous flush when the display is double buffered.
	 *
	 * When the call to {@link #flush()} performs a synchronous flush, this method should stay empty because there is
	 * nothing to wait (implementation).
	 *
	 * When the call to {@link #flush()} performs an asynchronous flush (to simulate the same behavior than embedded
	 * side): this method is useful to wait the end of this asynchronous flush.
	 *
	 * If the display does not have a <code>backBuffer</code> (not double buffered), nothing is done. This method should
	 * stay empty.
	 */
	public void waitFlush() {
		this.flushThread.waitFlush();
	}

	/**
	 * Refreshed the display.
	 */
	private void refresh() {
		this.refreshStrategy.refresh(this);
		this.hasFlushed = true;
	}

	private void initializeBoundingBox(Rectangle rectangle) {
		rectangle.setX1(Integer.MAX_VALUE);
		rectangle.setY1(Integer.MAX_VALUE);
		rectangle.setX2(Integer.MIN_VALUE);
		rectangle.setY2(Integer.MIN_VALUE);
	}

	/**
	 * Flushes the display with the help of the given rectangles.
	 *
	 * @param rectangles
	 *            the modified rectangles.
	 */
	public void flush(Rectangle[] rectangles) {
		this.drawnRectangles.clear();
		this.drawnRectangles.addAll(Arrays.asList(rectangles));
		this.bufferPolicy.flush(this, rectangles);
	}

	/**
	 * Restores (copies) the given rectangular region from old back buffer to the graphics context's current buffer.
	 *
	 * @param rectangle
	 *            the rectangle to restore.
	 */
	public void restore(Rectangle rectangle) {
		int x1 = rectangle.getX1();
		int y1 = rectangle.getY1();
		int width = rectangle.getWidth();
		int height = rectangle.getHeight();
		if (this.debugRestored) {
			int x2 = rectangle.getX2();
			int y2 = rectangle.getY2();
			this.restoredRectangles.add(new Rectangle(x1, y1, x2, y2));
			this.displayWidget.repaint();
		}
		this.bufferPolicy.getBackBuffer().drawImage(this.bufferPolicy.getFrontBuffer(), x1, y1, width, height, x1, y1,
				width, height);
	}

	/**
	 * Tells whether the buffer is clean (if nothing is drawn since flush) or not.
	 *
	 * @return <code>true</code> if nothing has been drawn since flush, <code>false</code> otherwise.
	 */
	public boolean isCleanBuffer() {
		return this.hasFlushed;
	}

	/**
	 * Simulates the embedded display's flush time when double buffering mode is enabled.
	 * <p>
	 * The buffer manager will wait this time before unlocking the caller to {@link #waitFlush()}.
	 * <p>
	 * The time is expressed in milliseconds. A negative or zero value (value) indicates the flush time is
	 * instantaneous. Typical values are between 5 and 10ms.
	 *
	 * @param ms
	 *            the flush time in milliseconds.
	 */
	public void setFlushTime(int ms) {
		this.flushTimeMs = ms;
	}

	/**
	 * Simulates the embedded display's refresh rate when double buffering mode is enabled.
	 * <p>
	 * When set, the widget is only allowed to initiate a flush action at every refresh tick. It is synchronized on the
	 * hardware display's periodic signal (tearing).
	 *
	 * @param ms
	 *            the refresh time in ms.
	 */
	public void setRefreshTime(int ms) {
		this.refreshRateTimeMs = ms;
	}

	/**
	 * Simulates the flush time.
	 */
	public void simulateFlushTime() {
		simulateWork(this.flushTimeMs);
	}

	private void simulateWork(double ms) {
		// cannot use Thread.sleep(): the scheduling perverts the timings
		long currentTimeMillis = System.currentTimeMillis();
		while (System.currentTimeMillis() - currentTimeMillis < ms) {
			Thread.yield();
		}
	}

	/**
	 * Called before a drawing is done.
	 *
	 * @param x1
	 *            the dirty region left x coordinate
	 * @param y1
	 *            the dirty region top y coordinate
	 * @param x2
	 *            the dirty region right y coordinate
	 * @param y2
	 *            the dirty region bottom y coordinate
	 * @param drawingNow
	 *            <code>true</code> if a drawing is following this call, <code>false</code> otherwise.
	 */
	public void newDrawingRegion(int x1, int y1, int x2, int y2, boolean drawingNow) {
		Rectangle rectangle = new Rectangle(x1, y1, x2, y2);
		synchronized (this.monitor) {
			if (this.hasFlushed) {
				this.restoredRectangles.clear();
			}
			this.hasFlushed = false;
			this.refreshStrategy.newDrawingRegion(this, rectangle, drawingNow);
		}
		updateDirtyRegion(rectangle);
	}

	private void updateDirtyRegion(Rectangle rectangle) {
		if (this.debugDirty) {
			Image currentBackBuffer = getCurrentBackBuffer();
			if (currentBackBuffer != this.dirtyRectangleBuffer) {
				this.dirtyRectangleBuffer = currentBackBuffer;
				initializeBoundingBox(this.dirtyRectangle);
			}
			if (updateDirtyRegionBoundingBox(rectangle)) {
				setDirtyRegion(rectangle);
			}
		}
	}

	private boolean updateDirtyRegionBoundingBox(Rectangle rectangle) {
		int x1 = rectangle.getX1();
		int x2 = rectangle.getX2();
		int y1 = rectangle.getY1();
		int y2 = rectangle.getY2();
		boolean result = false;
		Rectangle dirtyRectangle = this.dirtyRectangle;
		if (dirtyRectangle.getX1() > x1) {
			dirtyRectangle.setX1(x1);
			result = true;
		}
		if (dirtyRectangle.getY1() > y1) {
			dirtyRectangle.setY1(y1);
			result = true;
		}
		if (dirtyRectangle.getX2() < x2) {
			dirtyRectangle.setX2(x2);
			result = true;
		}
		if (dirtyRectangle.getY2() < y2) {
			dirtyRectangle.setY2(y2);
			result = true;
		}
		return result;
	}

	private void setDirtyRegion(Rectangle rectangle) {
		this.bufferPolicy.getBackBuffer().fillTransparentRectangle(rectangle.getX1(), rectangle.getY1(),
				rectangle.getWidth(), rectangle.getHeight(), this.dirtyColor);
	}

	@Override
	public int getWidth() {
		return getCurrentBackBuffer().getWidth();
	}

	@Override
	public int getHeight() {
		return getCurrentBackBuffer().getHeight();
	}

	@Override
	public void setPixels(int[] pixels) {
		getCurrentBackBuffer().setPixels(pixels);
	}

	@Override
	public int readPixel(int x, int y) {
		return getCurrentBackBuffer().readPixel(x, y);
	}

	@Override
	public void drawPixel(int x, int y, int color) {
		getCurrentBackBuffer().drawPixel(x, y, color);
	}

	@Override
	public void fillRectangle(int x, int y, int width, int height, int color) {
		getCurrentBackBuffer().fillRectangle(x, y, width, height, color);
	}

	@Override
	public Image getTransparentImage(byte alpha) {
		throw new IllegalStateException();
	}

	@Override
	public Object getRAWImage() {
		return getCurrentBackBuffer().getRAWImage();
	}

	@Override
	public void getPixels(int[] pixels) {
		getCurrentBackBuffer().getPixels(pixels);
	}

	@Override
	public void fillTransparentRectangle(int x, int y, int width, int height, int argbColor) {
		getCurrentBackBuffer().fillTransparentRectangle(x, y, width, height, argbColor);
	}

	@Override
	public void fillOval(int x, int y, int width, int height, int color) {
		getCurrentBackBuffer().fillOval(x, y, width, height, color);
	}

	@Override
	public void drawTransparentImage(Image image, byte alpha, int sx, int sy, int sWidth, int sHeight, int dx, int dy,
			int dWidth, int dHeight) {
		getCurrentBackBuffer().drawTransparentImage(image, alpha, sx, sy, sWidth, sHeight, dx, dy, dWidth, dHeight);
	}

	@Override
	public void drawString(String s, int x, int y, int textColor, int backColor) {
		getCurrentBackBuffer().drawString(s, x, y, textColor, backColor);
	}

	@Override
	public void drawRectangle(int x, int y, int width, int height, int color) {
		getCurrentBackBuffer().drawRectangle(x, y, width, height, color);
	}

	@Override
	public void drawLine(int x1, int y1, int x2, int y2, int color) {
		getCurrentBackBuffer().drawLine(x1, y1, x2, y2, color);
	}

	@Override
	public void drawImage(Image image, int sx, int sy, int sWidth, int sHeight, int dx, int dy, int dWidth,
			int dHeight) {
		getCurrentBackBuffer().drawImage(image, sx, sy, sWidth, sHeight, dx, dy, dWidth, dHeight);
	}

	@Override
	public void setTransparentMode(boolean transparent) {
		getCurrentBackBuffer().setTransparentMode(transparent);
	}

	@Override
	public void crop(Image mask) throws IllegalArgumentException {
		getCurrentBackBuffer().crop(mask);
	}

	@Override
	public void dispose() {
		this.bufferPolicy.dispose();
		this.flushThread.dispose();
	}

	/**
	 * Third-party thread that manages the embedded display flush time and tearing signal.
	 */
	class FlushThread extends Thread {

		private final Object flushMonitor;
		private final Object waitFlushMonitor;
		private boolean wakedUp;
		private boolean isRunning;

		private FlushThread() {
			this.flushMonitor = new Object();
			this.waitFlushMonitor = new Object();
			this.isRunning = true;
		}

		/**
		 * Wakes-up the thread to launch a flush action.
		 */
		private void wakeup() {
			synchronized (this.flushMonitor) {
				this.wakedUp = true;

				this.flushMonitor.notify();
			}
		}

		/**
		 * Waits until the end of previous flush action.
		 */
		private void waitFlush() {
			synchronized (this.waitFlushMonitor) {
				while (this.isRunning && this.wakedUp) {
					try {
						this.waitFlushMonitor.wait();
					} catch (InterruptedException e) {
						Thread.currentThread().interrupt();
					}
				}
			}
		}

		/**
		 * Stops the thread.
		 */
		private void dispose() {
			this.isRunning = false;
			synchronized (this.flushMonitor) {
				this.flushMonitor.notifyAll();
			}
			synchronized (this.waitFlushMonitor) {
				this.waitFlushMonitor.notifyAll();
			}
		}

		@Override
		public void run() {
			while (this.isRunning) {

				// wait a request of flush action
				synchronized (this.flushMonitor) {
					while (this.isRunning && !this.wakedUp) {
						try {
							this.flushMonitor.wait();
						} catch (InterruptedException e) {
							Thread.currentThread().interrupt();
							return;
						}
					}

					if (this.isRunning) {
						// flush & simulate the embedded display flush time
						refresh();

						// unlock caller to waitFlush()
						this.wakedUp = false;
						synchronized (this.waitFlushMonitor) {
							this.waitFlushMonitor.notifyAll();
						}
					}

					double refreshRateTime = DisplayBufferManager.this.refreshRateTimeMs;
					if (refreshRateTime > 0 /* prevent div by 0 */) {
						// get time to reach next tearing signal
						// sleep to reach the next tearing tick
						simulateWork(refreshRateTime);
					}
				}
			}
		}
	}

	private static int getColorProperty(String name) {
		return Long.getLong(name, FULLY_TRANSPARENT).intValue();
	}

	private static boolean isSetColorProperty(String name) {
		return Long.getLong(name) != null;
	}

	private static boolean isFullyTransparentColor(int color) {
		int alpha = (color & ALPHA_MASK) >>> ALPHA_SHIFT;
		return alpha == 0x0;
	}

	private static int getForegroundColor(int color) {
		if (isFullyTransparentColor(color)) {
			return FULLY_TRANSPARENT;
		} else {
			int red = (color & RED_MASK) >>> RED_SHIFT;
			int green = (color & GREEN_MASK) >>> GREEN_SHIFT;
			int blue = (color & BLUE_MASK) >>> BLUE_SHIFT;
			int light = (int) (red * RED_LIGHT_FACTOR + green * GREEN_LIGHT_FACTOR + blue * BLUE_LIGHT_FACTOR);
			if (light > LIGHT_COLOR_LIMIT) {
				return BLACK_OPAQUE;
			} else {
				return WHITE_OPAQUE;
			}
		}
	}

}
