/*
 * Java
 *
 * Copyright 2019-2021 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.util.message.basic;

import ej.util.message.Level;
import ej.util.message.MessageBuilder;
import ej.util.message.MessageLogger;

/**
 * A message logger delegates the messages build to a {@link MessageBuilder} and prints them to the standard output
 * stream when the message level is higher that logger threshold.
 *
 * @see System#out
 */
public class FilterMessageLogger implements MessageLogger {

	private static final char FINEST = 10;
	private static final char FINER = 20;
	private static final char FINE = 30;
	private static final char INFO = 40;
	private static final char CONFIG = 50;
	private static final char WARNING = 60;
	private static final char SEVERE = 70;
	/**
	 * Print no messages.
	 */
	public static final char OFF = 0xFF;
	/**
	 * Print all message.
	 */
	public static final char ALL = 0;

	/**
	 * An instance of FilterMessageLogger.
	 */
	public static final FilterMessageLogger INSTANCE = new FilterMessageLogger();

	private MessageBuilder builder;
	private char printLevel;

	/**
	 * Creates an instance using {@link BasicMessageBuilder#INSTANCE} to build the messages.
	 *
	 */
	public FilterMessageLogger() {
		this(BasicMessageBuilder.INSTANCE);
	}

	/**
	 * Creates an instance passing the message builder used to build the messages. Sets the print level to
	 * {@link Level#INFO}.
	 *
	 * @param builder
	 *            the {@link MessageBuilder} to use, cannot be <code>null</code>.
	 * @throws NullPointerException
	 *             if the given builder is <code>null</code>.
	 */
	public FilterMessageLogger(MessageBuilder builder) {
		this(builder, Level.INFO);
	}

	/**
	 * Creates an instance passing the message builder used to build the messages.
	 *
	 * @param builder
	 *            the {@link MessageBuilder} to use, cannot be <code>null</code>.
	 * @param level
	 *            the print level threshold.
	 * @throws NullPointerException
	 *             if the given builder is <code>null</code>.
	 */
	public FilterMessageLogger(MessageBuilder builder, char level) {
		if (builder == null) {
			throw new NullPointerException();
		}
		this.builder = builder;
		setLevel(level);
	}

	@Override
	public MessageBuilder getMessageBuilder() {
		return this.builder;
	}

	@Override
	public void setMessageBuilder(MessageBuilder builder) {
		if (builder == null) {
			throw new NullPointerException();
		}
		this.builder = builder;
	}

	/**
	 * Sets the level of the logger.
	 *
	 * The level should be:
	 * <ul>
	 * <li>{@link FilterMessageLogger#OFF}.</li>
	 * <li>{@link FilterMessageLogger#ALL}.</li>
	 * <li>Any {@link Level}.</li>
	 * </ul>
	 *
	 * @param level
	 *            the level to use.
	 * @throws IllegalArgumentException
	 *             if the given level is not valid.
	 */
	public void setLevel(char level) throws IllegalArgumentException {
		this.printLevel = getLevelId(level);
	}

	/**
	 * Gets the level of the logger.
	 *
	 * The level will be:
	 * <ul>
	 * <li>{@link FilterMessageLogger#OFF}</li>
	 * <li>{@link FilterMessageLogger#ALL}</li>
	 * <li>Any {@link Level}</li>
	 * </ul>
	 *
	 * @return the logger level.
	 */
	public char getLevel() {
		return getLevelFromId(this.printLevel);
	}

	@Override
	public void log(char level, String category, int id) {
		if (shouldPrint(level)) {
			System.out.println(this.builder.buildMessage(level, category, id));
		}
	}

	@Override
	public void log(char level, String category, int id, Throwable t) {
		if (shouldPrint(level)) {
			System.out.println(this.builder.buildMessage(level, category, id, t));
		}
	}

	@Override
	public void log(char level, String category, int id, Object... arguments) {
		if (shouldPrint(level)) {
			System.out.println(this.builder.buildMessage(level, category, id, arguments));
		}

	}

	@Override
	public void log(char level, String category, int id, Throwable t, Object... arguments) {
		if (shouldPrint(level)) {
			System.out.println(this.builder.buildMessage(level, category, id, t, arguments));
		}
	}

	private boolean shouldPrint(char level) {
		return this.printLevel <= getLevelId(level);
	}

	private char getLevelId(char level) {
		switch (level) {
		case Level.SEVERE:
			return SEVERE;
		case Level.WARNING:
			return WARNING;
		case Level.CONFIG:
			return CONFIG;
		case Level.INFO:
			return INFO;
		case Level.FINE:
			return FINE;
		case Level.FINER:
			return FINER;
		case Level.FINEST:
			return FINEST;
		case OFF:
			return OFF;
		case ALL:
			return ALL;
		}
		throw new IllegalArgumentException();
	}

	private char getLevelFromId(char level) {
		switch (level) {
		case SEVERE:
			return Level.SEVERE;
		case WARNING:
			return Level.WARNING;
		case CONFIG:
			return Level.CONFIG;
		case INFO:
			return Level.INFO;
		case FINE:
			return Level.FINE;
		case FINER:
			return Level.FINER;
		case FINEST:
			return Level.FINEST;
		case OFF:
			return OFF;
		case ALL:
			return ALL;
		}
		throw new IllegalArgumentException();
	}
}
