/*
 * Java
 *
 * Copyright 2025 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.basictool;

import java.io.EOFException;
import java.io.IOException;
import java.io.InputStream;

/**
 * Provides utility methods to manipulate streams.
 */
public class StreamUtils {

	private StreamUtils() {
		// private constructor
	}

	/**
	 * Reads an exact number of bytes from the given input stream.
	 *
	 * @param inputStream
	 *            the input stream to read from.
	 * @param array
	 *            the array that will contain the data read.
	 * @param offset
	 *            the offset in the array of the first byte read.
	 * @param size
	 *            the number of bytes to read.
	 * @throws EOFException
	 *             if the end of stream is reached before reading the given number of bytes.
	 * @throws IOException
	 *             if an error occurred when reading from the input stream.
	 */
	public static void readFully(InputStream inputStream, byte[] array, int offset, int size) throws IOException {
		int end = offset + size;
		while (offset < end) {
			int bytesRead = inputStream.read(array, offset, end - offset);
			if (bytesRead == -1) {
				throw new EOFException();
			}
			offset += bytesRead;
		}
	}

	/**
	 * Skips an exact number of bytes from the given input stream.
	 *
	 * @param inputStream
	 *            the input stream to skip from.
	 * @param size
	 *            the number of bytes to skip.
	 * @throws EOFException
	 *             if the end of stream is reached before skipping the given number of bytes.
	 * @throws IOException
	 *             if an error occurred when reading from the input stream.
	 */
	public static void skipFully(InputStream inputStream, long size) throws IOException {
		while (size > 0) {
			long bytesSkipped = inputStream.skip(size);
			if (bytesSkipped == 0) {
				if (inputStream.read() == -1) {
					throw new EOFException();
				}
				size--;
			} else {
				size -= bytesSkipped;
			}
		}
	}
}
