/*
 * Java
 *
 * Copyright 2024 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package com.microej.kf.util.security;

import java.util.List;
import java.util.Objects;

import ej.annotation.Nullable;
import ej.basictool.map.PackedMap;

/**
 * This class represents the permission attributes retrieved by the parser.
 */
public class FeaturePolicyPermission {

	private final String permissionClassName;
	private final PackedMap<String, List<String>> nameActionsMap;

	/**
	 * Instantiates a new feature policy permission.
	 *
	 * @param permissionClassName the permission class name
	 * @param nameActionsMap      the name actions map
	 */
	public FeaturePolicyPermission(String permissionClassName, PackedMap<String, List<String>> nameActionsMap) {
		this.permissionClassName = permissionClassName;
		this.nameActionsMap = nameActionsMap;
	}

	/**
	 * Gets the permission class name.
	 *
	 * @return the permission class name
	 */
	public String getPermissionClassName() {
		return this.permissionClassName;
	}

	/**
	 * Gets the name actions map.
	 *
	 * @return the name actions map
	 */
	public PackedMap<String, List<String>> getNameActionsMap() {
		return this.nameActionsMap;
	}

	/**
	 * Equals.
	 *
	 * @param o the o
	 * @return true, if successful
	 */
	@Override
	public boolean equals(@Nullable Object o) {
		if (!(o instanceof FeaturePolicyPermission)) {
			return false;
		}

		FeaturePolicyPermission otherObject = (FeaturePolicyPermission) o;

		return this.permissionClassName.equals(otherObject.permissionClassName)
				&& (this.nameActionsMap.size() == otherObject.nameActionsMap.size()
				&& this.nameActionsMap.values().containsAll(otherObject.nameActionsMap.values()));
	}

	@Override
	public int hashCode() {
		return Objects.hash(this.permissionClassName, this.nameActionsMap.size(), this.nameActionsMap.values());
	}
}
