/*
 * Java
 *
 * Copyright 2022 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package com.microej.microvg.test;

import org.junit.AfterClass;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;

import ej.microui.display.Colors;
import ej.microui.display.Display;
import ej.microui.display.GraphicsContext;
import ej.microui.display.Painter;
import ej.microvg.BlendMode;
import ej.microvg.LinearGradient;
import ej.microvg.Matrix;
import ej.microvg.VectorFont;
import ej.microvg.VectorGraphicsPainter;
import ej.microvg.VectorGraphicsPainter.Direction;

/**
 * Tests kerning.
 */
public class TestFontComplexLayoutOnCircle {

	private static final String NOTOSANSARABIC_FONT = "/fonts/NotoSansArabic-reduced.ttf";

	private static final String ARABIC_STRING = "الطريق أخضر ومظلل";

	/**
	 * Starts MicroUI.
	 */
	@BeforeClass
	public static void pre() {
		TestUtilities.startMicroUI();
	}

	/**
	 * Stops MicroUI.
	 */
	@AfterClass
	public static void post() {
		TestUtilities.stopMicroUI();
	}

	/**
	 * Resets the content of the screen to black.
	 */
	@Before
	public static void preTest() {
		TestUtilities.clearScreen();
	}

	/**
	 * Test drawing an arabic string on a circle
	 *
	 */
	@Test
	public void testDrawStringLayoutOnCircle() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();
		VectorFont font = VectorFont.loadFont(NOTOSANSARABIC_FONT, true);

		String string = ARABIC_STRING;

		int fontSize = 30;
		int x = display.getWidth() / 2;
		int y = display.getHeight() / 2;
		int diameter = 250;

		g.setColor(Colors.RED);
		Painter.drawCircle(g, x - diameter / 2, y - diameter / 2, diameter);

		g.setColor(Colors.PURPLE);
		Matrix matrix = new Matrix();

		matrix.setTranslate(x, y);
		VectorGraphicsPainter.drawStringOnCircle(g, string, font, fontSize, matrix, diameter / 2,
				Direction.COUNTER_CLOCKWISE);

		g.setColor(Colors.YELLOW);
		VectorGraphicsPainter.drawStringOnCircle(g, string, font, fontSize, matrix, diameter / 2, Direction.CLOCKWISE);

		display.flush();

		float delta = TestUtilities.isOnAndroid() ? 0.08f : 0.05f;
		TestUtilities.compareDisplay(TestUtilities.getImageContextualName("testDrawStringLayoutOnCircle"), Colors.BLACK,
				0, 0, display.getWidth(), display.getHeight(), delta);
	}

	/**
	 * Test drawing an arabic string on a circle with a gradient
	 *
	 */
	@Test
	public void testDrawStringLayoutOnCircleGradient() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();
		VectorFont font = VectorFont.loadFont(NOTOSANSARABIC_FONT, true);

		String string = ARABIC_STRING;

		int fontSize = 30;
		int x = display.getWidth() / 2;
		int y = display.getHeight() / 2;
		int diameter = 250;

		g.setColor(Colors.RED);
		Painter.drawCircle(g, x - diameter / 2, y - diameter / 2, diameter);

		// Colors must be provided with alpha channel
		int[] colors = { TestUtilities.getColorWithAlpha(Colors.BLUE, GraphicsContext.OPAQUE),
				TestUtilities.getColorWithAlpha(Colors.YELLOW, GraphicsContext.OPAQUE),
				TestUtilities.getColorWithAlpha(Colors.GREEN, GraphicsContext.OPAQUE) };
		float[] stops = { 0f, 0.5f, 1f };

		LinearGradient gradient = new LinearGradient(0, -diameter / 2, 0, diameter / 2, colors, stops);

		Matrix matrix = new Matrix();

		matrix.setTranslate(x, y);
		VectorGraphicsPainter.drawGradientStringOnCircle(g, string, font, fontSize, matrix, gradient, diameter / 2,
				Direction.COUNTER_CLOCKWISE, GraphicsContext.OPAQUE, BlendMode.SRC_OVER, 0);

		VectorGraphicsPainter.drawGradientStringOnCircle(g, string, font, fontSize, matrix, gradient, diameter / 2,
				Direction.CLOCKWISE, GraphicsContext.OPAQUE, BlendMode.SRC_OVER, 0);

		display.flush();

		float delta = TestUtilities.isOnAndroid() ? 0.06f : 0.05f;
		TestUtilities.compareDisplay(TestUtilities.getImageContextualName("testDrawStringLayoutOnCircleGradient"),
				Colors.BLACK, 0, 0, display.getWidth(), display.getHeight(), delta);
	}

}
