/*
 * Java
 *
 * Copyright 2023 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package com.microej.microvg.test;

import org.junit.AfterClass;
import org.junit.Assert;
import org.junit.BeforeClass;
import org.junit.Test;

import ej.bon.Constants;
import ej.microui.MicroUI;
import ej.microui.MicroUIException;
import ej.microui.display.BufferedImage;
import ej.microui.display.Display;
import ej.microui.display.GraphicsContext;
import ej.microvg.BufferedVectorImage;
import ej.microvg.Matrix;
import ej.microvg.Path;
import ej.microvg.VectorGraphicsException;
import ej.microvg.VectorGraphicsPainter;

/**
 * Tests various features of BufferedVectorImage.
 */
public class TestBufferedVectorImage {

	/**
	 * The test {@link #testParalledDrawingInClosedImage()} maybe too long to execute and may throw a timeout exception.
	 * The following constants may help to adjust the execution time.
	 * <p>
	 * The number of loops must be large enough to let the MicroJVM performing a thread scheduling.
	 */
	private static final String TestParalledDrawingInClosedImageNbLoopsSim = "TestBufferedVectorImage.testParalledDrawingInClosedImage.sim";
	private static final String TestParalledDrawingInClosedImageNbLoopsEmb = "TestBufferedVectorImage.testParalledDrawingInClosedImage.emb";

	/**
	 * The test {@link #testParalledDrawingClosedImage()} maybe too long to execute and may throw a timeout exception.
	 *
	 * @see #TestParalledDrawingInClosedImageNbLoopsSim
	 */
	private static final String TestParalledDrawingClosedImageNbLoopsSim = "TestBufferedVectorImage.testParalledDrawingClosedImage.sim";
	private static final String TestParalledDrawingClosedImageNbLoopsEmb = "TestBufferedVectorImage.testParalledDrawingClosedImage.emb";

	/**
	 * Initializes MicroUI.
	 */
	@BeforeClass
	public static void setUpBeforeClass() {
		MicroUI.start();
	}

	/**
	 * Stops MicroUI.
	 */
	@AfterClass
	public static void tearDownAfterClass() {
		MicroUI.stop();
	}

	/**
	 * Tests that creating and closing many BufferedVectorImage in succession does not cause a memory leak.
	 */
	@Test
	public void testMemoryLeak() {
		for (int i = 0; i < 1000; i++) {
			@SuppressWarnings("resource")
			BufferedVectorImage bvi = new BufferedVectorImage(1, 1);
			bvi.close();
		}
	}

	/**
	 * Tests that drawing into a closed BufferedVectorImage throws an exception.
	 */
	@Test
	public void testDrawingInClosedImage() {
		@SuppressWarnings("resource")
		BufferedVectorImage bvi = new BufferedVectorImage(100, 100);
		GraphicsContext gc = bvi.getGraphicsContext();
		bvi.close();
		try {
			VectorGraphicsPainter.fillPath(gc, newPath(), new Matrix());
			Assert.fail("exception not thrown"); //$NON-NLS-1$
		} catch (MicroUIException e) {
			Assert.assertEquals("check exception value", MicroUIException.RESOURCE_CLOSED, e.getErrorCode()); //$NON-NLS-1$
		}
	}

	/**
	 * Tests that drawing a closed BufferedVectorImage throws an exception.
	 */
	@Test
	public void testDrawingClosedImage() {
		@SuppressWarnings("resource")
		BufferedVectorImage bvi = new BufferedVectorImage(100, 100);
		GraphicsContext gc = bvi.getGraphicsContext();
		VectorGraphicsPainter.fillPath(gc, newPath(), new Matrix());
		bvi.close();

		try {
			VectorGraphicsPainter.drawImage(Display.getDisplay().getGraphicsContext(), bvi, new Matrix());
			Assert.fail("exception not thrown"); //$NON-NLS-1$
		} catch (VectorGraphicsException e) {
			Assert.assertEquals("check exception value", VectorGraphicsException.RESOURCE_CLOSED, e.getErrorCode()); //$NON-NLS-1$
		}
	}

	/**
	 * Tests that drawing into a closed BufferedVectorImage in another thread does not alterate the image heap: the heap
	 * must be unchanged at the end of the test and must not throw any error when creating / closing the images.
	 */
	@Test
	public void testParalledDrawingInClosedImage() {

		final int nbLoops = TestUtilities.isOnSimulator() ? Constants.getInt(TestParalledDrawingInClosedImageNbLoopsSim)
				: Constants.getInt(TestParalledDrawingInClosedImageNbLoopsEmb);

		final Path path = newPath();
		final Matrix matrix = new Matrix();
		final GraphicsContext[] gc = new GraphicsContext[1];

		int nbAllocatedImage = allocUntilOOM();

		// thread that creates and closes a buffered vector image
		new Thread(new Runnable() {
			@Override
			public void run() {

				try {
					for (int i = 0; i < nbLoops; i++) {
						try (BufferedVectorImage bvi = new BufferedVectorImage(100, 100)) {
							gc[0] = bvi.getGraphicsContext();
						}
					}
				} catch (Throwable e) {
					// NPE, OOM, MicroUI exception, etc.: test fails
					Assert.fail("exception during image creation: " + e.getMessage()); //$NON-NLS-1$
				}
				gc[0] = null;
			}
		}).start();

		// ensure first thread is running
		try {
			Thread.sleep(500);
		} catch (InterruptedException e1) {
			e1.printStackTrace();
		}

		// thread that draws in the buffered vector image created by the other thread
		int drawings = 0;
		while (null != gc[0]) {
			try {
				VectorGraphicsPainter.fillPath(gc[0], path, matrix);
				++drawings;
			} catch (Throwable e) {
				// NPE, OOM, MicroUI exception, etc., ignore error
			}
		}

		Assert.assertNotEquals(
				"test has run (at least one drawing has been performed); on error, adjust the BON Constant(s) TestParalledDrawingInClosedImageNbLoopsSim|Emb", //$NON-NLS-1$
				0, drawings);
		Assert.assertEquals("image heap is unchanged", nbAllocatedImage, allocUntilOOM()); //$NON-NLS-1$
	}

	/**
	 * Tests that drawing a closed BufferedVectorImage in another thread does not alterate the image heap: the heap must
	 * be unchanged at the end of the test and must not throw any error when creating / closing the images.
	 */
	@Test
	public void testParalledDrawingClosedImage() {

		final int nbLoops = TestUtilities.isOnSimulator() ? Constants.getInt(TestParalledDrawingClosedImageNbLoopsSim)
				: Constants.getInt(TestParalledDrawingClosedImageNbLoopsEmb);

		final Matrix matrix = new Matrix();
		final BufferedVectorImage[] images = new BufferedVectorImage[1];

		int nbAllocatedImage = allocUntilOOM();

		// thread that creates and closes a buffered vector image
		new Thread(new Runnable() {
			@Override
			public void run() {
				try {
					for (int i = 0; i < nbLoops; i++) {
						try (BufferedVectorImage bvi = new BufferedVectorImage(100, 100)) {
							images[0] = bvi;
						}
					}
				} catch (Throwable e) {
					// NPE, OOM, MicroUI exception, etc.: test fails
					Assert.fail("exception during image creation: " + e.getMessage()); //$NON-NLS-1$
				}
				images[0] = null;
			}
		}).start();

		// ensure first thread is running
		try {
			Thread.sleep(500);
		} catch (InterruptedException e1) {
			e1.printStackTrace();
		}

		// thread that draws in the buffered vector image created by the other thread
		int drawings = 0;
		GraphicsContext gc = Display.getDisplay().getGraphicsContext();
		while (null != images[0]) {
			try {
				VectorGraphicsPainter.drawImage(gc, images[0], matrix);
				++drawings;
			} catch (Throwable e) {
				// NPE, OOM, MicroUI exception, etc., ignore error
			}
		}

		Assert.assertNotEquals("test has run (at least one drawing)", 0, drawings); //$NON-NLS-1$
		Assert.assertEquals("image heap is unchanged", nbAllocatedImage, allocUntilOOM()); //$NON-NLS-1$
	}

	private int allocUntilOOM() {

		class ImageList {
			ImageList next;
			BufferedImage image;

			ImageList() {
				this.image = new BufferedImage(2, 2);
			}
		}

		ImageList first = new ImageList();
		ImageList list = first;
		int cpt = 0;

		try {
			while (true) {
				++cpt;
				list.next = new ImageList();
				list = list.next;
			}
		} catch (MicroUIException e) {
			while (first != null) {
				first.image.close();
				first = first.next;
			}
		}

		return cpt;
	}

	private Path newPath() {
		Path path = new Path();
		path.lineTo(100, 0);
		path.lineTo(100, 100);
		path.lineTo(0, 100);
		path.close();
		return path;
	}
}
