/*
 * Java
 *
 * Copyright 2008-2019 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.fp;

import java.util.LinkedList;
import java.util.NoSuchElementException;

/**
 * The events queue is a mechanism to store some events in a linked list. These events will be read by a consumer later.
 * <p>
 * The events are some <i>integers</i>. The format of this integer is producer / consumer dependent.
 * <p>
 * The device holds only one queue. This queue is known and used by the Front Panel plugin and by the MicroEJ library
 * 'MicroUI'.
 */
public class EventQueue {

	// events queue
	private final LinkedList<Integer> queue;

	/**
	 * Creates a new event queue.
	 */
	public EventQueue() {
		this.queue = new LinkedList<>();
	}

	/**
	 * Adds the events in the queue of events, respecting the array order.
	 *
	 * @param events
	 *            the events to add.
	 */
	public synchronized void enqueudEvents(int[] events) {
		int l = events.length;
		for (int i = -1; ++i < l;) {
			add(events[i]);
		}
		notifyAll();
	}

	/**
	 * Adds an event in the queue of events.
	 *
	 * @param event
	 *            the event to add.
	 */
	public synchronized void enqueudEvent(int event) {
		add(event);
		notifyAll();
	}

	/**
	 * Reads an event in the queue of events. If there is no event, the thread is waiting while there is no event.
	 *
	 * @return the event read.
	 * @throws InterruptedException
	 *             if any thread interrupted the current thread before or while the current thread was waiting for a
	 *             notification.
	 */
	public synchronized int getEvent() throws InterruptedException {
		while (true) {
			try {
				return this.queue.remove().intValue();
			} catch (NoSuchElementException e) {
				wait();
			}
		}
	}

	private void add(int event) {
		this.queue.offer(Integer.valueOf(event));
	}

}
