/*
 *	Java
 *
 *	Copyright 2015-2025 MicroEJ Corp.
 *	Use of this source code is governed by a BSD-style license that can be found with this software.
 */

package ej.drawing;

import ej.microui.display.Font;
import ej.microui.display.GraphicsContext;
import ej.microui.display.Image;
import ej.microui.display.RenderableString;

/**
 * The <code>TransformPainter</code> class offers a set of static methods to render elements (image, renderable
 * character) with a transformation.
 * <p>
 * The transformation (rotate, scale etc.) is applied during the drawing: the source element is not updated (for
 * instance the source image is not rotated: this is its rendering on the graphics context which is rotated).
 */
public class TransformPainter {

	/**
	 * Specify the flip to apply when drawing an image.
	 *
	 * @see #drawFlippedImage(GraphicsContext, Image, int, int, Flip)
	 * @see #drawFlippedImage(GraphicsContext, Image, int, int, Flip, int)
	 */
	public enum Flip {

		/**
		 * Resets the flip action (0 degrees).<br>
		 * This is the default action.
		 */
		FLIP_NONE,

		/**
		 * Flips anticlockwise at 90 degrees.
		 */
		FLIP_90,

		/**
		 * Flips at 180 degrees.
		 */
		FLIP_180,

		/**
		 * Flips anticlockwise at 270 degrees.
		 */
		FLIP_270,

		/**
		 * Applies a mirror.
		 */
		FLIP_MIRROR,

		/**
		 * Applies a mirror and flip anticlockwise at 90 degrees.
		 */
		FLIP_MIRROR_90,

		/**
		 * Applies a mirror and flip at 180 degrees.
		 */
		FLIP_MIRROR_180,

		/**
		 * Applies a mirror and flip anticlockwise at 270 degrees.
		 */
		FLIP_MIRROR_270;
	}

	private TransformPainter() {
		// static methods only
	}

	/**
	 * Draws the image in the graphics context at given anchor top-left position, using the given flip.
	 * <p>
	 * Equivalent to calling {@link #drawFlippedImage(GraphicsContext, Image, int, int, Flip, int)} with
	 * {@link GraphicsContext#OPAQUE} as alpha.
	 *
	 * @param gc
	 *            the graphics context where to render the drawing.
	 * @param image
	 *            the image to draw
	 * @param x
	 *            the x coordinate of the image reference anchor top-left point
	 * @param y
	 *            the y coordinate of the image reference anchor top-left point
	 * @param flip
	 *            the flip to apply.
	 */
	public static void drawFlippedImage(GraphicsContext gc, Image image, int x, int y, Flip flip) {
		throw new RuntimeException();
	}

	/**
	 * Draws the image in the graphics context at given anchor top-left position, using the given flip and opacity
	 * level.
	 * <p>
	 * In addition with {@link #drawFlippedImage(GraphicsContext, Image, int, int, Flip)}, this method allows to specify
	 * the global opacity value to apply during the image rendering.
	 *
	 * @param gc
	 *            the graphics context where to render the drawing.
	 * @param image
	 *            the image to draw
	 * @param x
	 *            the x coordinate of the image reference anchor top-left point
	 * @param y
	 *            the y coordinate of the image reference anchor top-left point
	 * @param flip
	 *            the flip to apply.
	 * @param alpha
	 *            the global opacity rendering value.
	 * @throws IllegalArgumentException
	 *             if the given alpha is not a value between {@link GraphicsContext#TRANSPARENT} and
	 *             {@link GraphicsContext#OPAQUE}.
	 */
	public static void drawFlippedImage(GraphicsContext gc, Image image, int x, int y, Flip flip, int alpha) {
		throw new RuntimeException();
	}

	/**
	 * Draws a deformed image at the given anchor top-left point.<br>
	 * <br>
	 * The deformed image is identified by its four corner points. These points are defined by the array of integer
	 * coordinates and they must respect the following order: first is the top-left corner, second is the top-right,
	 * third is the bottom-right and fourth is the bottom-left.<br>
	 * <br>
	 * Examples with img an image and imgWidth and imgHeight its size.<br>
	 * <ul>
	 * <li>To draw normal img, the array should be : {0, 0, imgWidth-1, 0, imgWidth-1, imgHeight-1, 0, imgHeight-1}.
	 * </li>
	 * <li>To draw img with a rotation clockwise by 90 degrees, the array should be : {imgHeight-1, 0, imgHeight-1,
	 * imgWidth-1, 0, imgWidth-1, 0, 0}.</li>
	 * <li>To draw img mirrored about the vertical axis, the array should be : {0, 0, -(imgWidth-1), 0, -(imgWidth-1),
	 * -(imgHeight-1), 0, -(imgHeight-1)}.</li>
	 * <li>To draw img with a double scale, the array should be : {0, 0, (imgWidth-1)*2, 0, (imgWidth-1)*2,
	 * (imgHeight-1)*2, 0, (imgHeight-1)*2}.</li>
	 * </ul>
	 *
	 * @param gc
	 *            the graphics context where to render the drawing.
	 * @param image
	 *            the image to draw
	 * @param x
	 *            the x coordinate of the anchor top-left point
	 * @param y
	 *            the y coordinate of the anchor top-left point
	 * @param xys
	 *            the array of coordinates : x1,y1,x2,y2,x3,y3,x4,y4.
	 * @throws IllegalArgumentException
	 *             if the @{code xys} length is different than 2*4.
	 */
	public static void drawDeformedImage(GraphicsContext gc, Image image, int x, int y, int[] xys) {
		throw new RuntimeException();
	}

	/**
	 * Draws the image applying the given rotation. This method uses the <code>bilinear</code> algorithm to render the
	 * content. This algorithm performs better rendering than <code>nearest neighbor</code> algorithm but it is slower
	 * to apply.
	 * <p>
	 * The rotation is specified by the center and the angle. The reference point is the graphical object top-left
	 * corner. The rotation point is relative to the current translation of the graphics context where the graphical
	 * object will be drawn.
	 * <p>
	 * To rotate an image on itself, use the following lines:
	 * <p>
	 * <code>
	 * int rx = x + image.getWidth() / 2;<br>
	 * int ry = y + image.getHeight() / 2;<br>
	 * TransformPainter.drawRotatedImageBilinear(gc, image, x, y, rx, ry, 78);<br>
	 * </code>
	 * <p>
	 * To rotate an image around a circle, use the following lines:
	 * <p>
	 * <code>
	 * for (int i = 0; i &lt; 360; i += 45) {<br>
	 *  TransformPainter.drawRotatedImageBilinear(gc, image, rx - diameter / 2, ry - diameter / 2, rx, ry, i);<br>
	 * }<br>
	 * </code>
	 * <p>
	 * Equivalent to calling {@link #drawRotatedImageBilinear(GraphicsContext, Image, int, int, int, int, float, int)}
	 * with {@link GraphicsContext#OPAQUE} as alpha.
	 *
	 * @param gc
	 *            the graphics context where to render the drawing.
	 * @param image
	 *            the image to draw.
	 * @param x
	 *            the x coordinate of the image reference anchor top-left point
	 * @param y
	 *            the y coordinate of the image reference anchor top-left point
	 * @param rx
	 *            the rotation center X coordinate.
	 * @param ry
	 *            the rotation center Y coordinate.
	 * @param angle
	 *            the rotation angle.
	 * @see #drawRotatedImageNearestNeighbor(GraphicsContext, Image, int, int, int, int, float)
	 */
	public static void drawRotatedImageBilinear(GraphicsContext gc, Image image, int x, int y, int rx, int ry,
			float angle) {
		throw new RuntimeException();
	}

	/**
	 * Draws the image applying the given rotation fixing the opacity level. This method uses the <code>bilinear</code>
	 * algorithm to render the content. This algorithm performs better rendering than <code>nearest neighbor</code>
	 * algorithm but it is slower to apply.
	 * <p>
	 * Some rotation samples are described in method comment
	 * {@link #drawRotatedImageBilinear(GraphicsContext, Image, int, int, int, int, float)}.
	 * <p>
	 * In addition with {@link #drawRotatedImageBilinear(GraphicsContext, Image, int, int, int, int, float)}, this
	 * method allows to specify the global opacity value to apply during the image rendering.
	 *
	 * @param gc
	 *            the graphics context where to render the drawing.
	 * @param image
	 *            the image to draw.
	 * @param x
	 *            the x coordinate of the image reference anchor top-left point
	 * @param y
	 *            the y coordinate of the image reference anchor top-left point
	 * @param rx
	 *            the rotation center X coordinate.
	 * @param ry
	 *            the rotation center Y coordinate.
	 * @param angle
	 *            the rotation angle.
	 * @param alpha
	 *            the global opacity rendering value.
	 * @throws IllegalArgumentException
	 *             if the given alpha is not a value between {@link GraphicsContext#TRANSPARENT} and
	 *             {@link GraphicsContext#OPAQUE}.
	 * @see #drawRotatedImageNearestNeighbor(GraphicsContext, Image, int, int, int, int, float, int)
	 */
	public static void drawRotatedImageBilinear(GraphicsContext gc, Image image, int x, int y, int rx, int ry,
			float angle, int alpha) {
		throw new RuntimeException();
	}

	/**
	 * Draws the image applying the given rotation. This method uses the <code>nearest neighbor</code> algorithm to
	 * render the content. This algorithm is faster than <code>bilinear</code> algorithm but its rendering is more
	 * simple.
	 * <p>
	 * Some rotation samples are described in method comment
	 * {@link #drawRotatedImageBilinear(GraphicsContext, Image, int, int, int, int, float)}.
	 * <p>
	 * Equivalent to calling
	 * {@link #drawRotatedImageNearestNeighbor(GraphicsContext, Image, int, int, int, int, float, int)} with
	 * {@link GraphicsContext#OPAQUE} as alpha.
	 *
	 * @param gc
	 *            the graphics context where render the drawing.
	 * @param image
	 *            the image to draw.
	 * @param x
	 *            the x coordinate of the image reference anchor top-left point
	 * @param y
	 *            the y coordinate of the image reference anchor top-left point
	 * @param rx
	 *            the rotation center X coordinate.
	 * @param ry
	 *            the rotation center Y coordinate.
	 * @param angle
	 *            the rotation angle.
	 * @see #drawRotatedImageBilinear(GraphicsContext, Image, int, int, int, int, float)
	 */
	public static void drawRotatedImageNearestNeighbor(GraphicsContext gc, Image image, int x, int y, int rx, int ry,
			float angle) {
		throw new RuntimeException();
	}

	/**
	 * Draws the image applying the given rotation and opacity level. This method uses the <code>nearest neighbor</code>
	 * algorithm to render the content. This algorithm is faster than <code>bilinear</code> algorithm but its rendering
	 * is more simple.
	 * <p>
	 * Some rotation samples are described in method comment
	 * {@link #drawRotatedImageBilinear(GraphicsContext, Image, int, int, int, int, float)}.
	 * <p>
	 * In addition with {@link #drawRotatedImageNearestNeighbor(GraphicsContext, Image, int, int, int, int, float)},
	 * this method allows to specify the global opacity value to apply during the image rendering.
	 *
	 * @param gc
	 *            the graphics context where render the drawing.
	 * @param image
	 *            the image to draw.
	 * @param x
	 *            the x coordinate of the image reference anchor top-left point
	 * @param y
	 *            the y coordinate of the image reference anchor top-left point
	 * @param rx
	 *            the rotation center X coordinate.
	 * @param ry
	 *            the rotation center Y coordinate.
	 * @param angle
	 *            the rotation angle.
	 * @param alpha
	 *            the global opacity rendering value.
	 * @throws IllegalArgumentException
	 *             if the given alpha is not a value between {@link GraphicsContext#TRANSPARENT} and
	 *             {@link GraphicsContext#OPAQUE}.
	 * @see #drawRotatedImageBilinear(GraphicsContext, Image, int, int, int, int, float, int)
	 */
	public static void drawRotatedImageNearestNeighbor(GraphicsContext gc, Image image, int x, int y, int rx, int ry,
			float angle, int alpha) {
		throw new RuntimeException();
	}

	/**
	 * Draws the character applying the given rotation. This method uses the <code>bilinear</code> algorithm to render
	 * the content. This algorithm performs better rendering than <code>nearest neighbor</code> algorithm but it is
	 * slower to apply.
	 * <p>
	 * This method only supports characters in the range U+0000 to U+FFFF. It does not handle characters with code
	 * points greater than U+FFFF, which are represented as a pair of char values (referred to as "surrogate pair").
	 * <p>
	 * Some rotation samples are described in method comment
	 * {@link #drawRotatedImageBilinear(GraphicsContext, Image, int, int, int, int, float)}.
	 * <p>
	 * Equivalent to calling
	 * {@link #drawRotatedCharBilinear(GraphicsContext, Font, char, int, int, int, int, float, int)} with
	 * {@link GraphicsContext#OPAQUE} as alpha.
	 *
	 * @param gc
	 *            the graphics context where to render the drawing.
	 * @param font
	 *            the font to use.
	 * @param c
	 *            the character to draw.
	 * @param x
	 *            the x coordinate of the character reference anchor top-left point
	 * @param y
	 *            the y coordinate of the character reference anchor top-left point
	 * @param rx
	 *            the rotation center X coordinate.
	 * @param ry
	 *            the rotation center Y coordinate.
	 * @param angle
	 *            the rotation angle.
	 * @see #drawRotatedCharNearestNeighbor(GraphicsContext, Font, char, int, int, int, int, float)
	 */
	public static void drawRotatedCharBilinear(GraphicsContext gc, Font font, char c, int x, int y, int rx, int ry,
			float angle) {
		throw new RuntimeException();
	}

	/**
	 * Draws the character applying the given rotation and opacicity level. This method uses the <code>bilinear</code>
	 * algorithm to render the content. This algorithm performs better rendering than <code>nearest neighbor</code>
	 * algorithm but it is slower to apply.
	 * <p>
	 * This method only supports characters in the range U+0000 to U+FFFF. It does not handle characters with code
	 * points greater than U+FFFF, which are represented as a pair of char values (referred to as "surrogate pair").
	 * <p>
	 * Some rotation samples are described in method comment
	 * {@link #drawRotatedImageBilinear(GraphicsContext, Image, int, int, int, int, float)}.
	 * <p>
	 * In addition with {@link #drawRotatedCharBilinear(GraphicsContext, Font, char, int, int, int, int, float)}, this
	 * method allows to specify the global opacity value to apply during the image rendering.
	 *
	 * @param gc
	 *            the graphics context where to render the drawing.
	 * @param font
	 *            the font to use.
	 * @param c
	 *            the character to draw.
	 * @param x
	 *            the x coordinate of the character reference anchor top-left point
	 * @param y
	 *            the y coordinate of the character reference anchor top-left point
	 * @param rx
	 *            the rotation center X coordinate.
	 * @param ry
	 *            the rotation center Y coordinate.
	 * @param angle
	 *            the rotation angle.
	 * @param alpha
	 *            the global opacity rendering value.
	 * @throws IllegalArgumentException
	 *             if the given alpha is not a value between {@link GraphicsContext#TRANSPARENT} and
	 *             {@link GraphicsContext#OPAQUE}.
	 * @see #drawRotatedCharNearestNeighbor(GraphicsContext, Font, char, int, int, int, int, float, int)
	 */
	public static void drawRotatedCharBilinear(GraphicsContext gc, Font font, char c, int x, int y, int rx, int ry,
			float angle, int alpha) {
		throw new RuntimeException();
	}

	/**
	 * Draws the character applying the given rotation. This method uses the <code>nearest neighbor</code> algorithm to
	 * render the content. This algorithm is faster than <code>bilinear</code> algorithm but its rendering is more
	 * simple.
	 * <p>
	 * This method only supports characters in the range U+0000 to U+FFFF. It does not handle characters with code
	 * points greater than U+FFFF, which are represented as a pair of char values (referred to as "surrogate pair").
	 * <p>
	 * Some rotation samples are described in method comment
	 * {@link #drawRotatedImageBilinear(GraphicsContext, Image, int, int, int, int, float)}.
	 * <p>
	 * Equivalent to calling
	 * {@link #drawRotatedCharNearestNeighbor(GraphicsContext, Font, char, int, int, int, int, float, int)} with
	 * {@link GraphicsContext#OPAQUE} as alpha.
	 *
	 * @param gc
	 *            the graphics context where render the drawing.
	 * @param font
	 *            the font to use.
	 * @param c
	 *            the character to draw.
	 * @param x
	 *            the x coordinate of the character reference anchor top-left point
	 * @param y
	 *            the y coordinate of the character reference anchor top-left point
	 * @param rx
	 *            the rotation center X coordinate.
	 * @param ry
	 *            the rotation center Y coordinate.
	 * @param angle
	 *            the rotation angle.
	 * @see #drawRotatedCharBilinear(GraphicsContext, Font, char, int, int, int, int, float)
	 */
	public static void drawRotatedCharNearestNeighbor(GraphicsContext gc, Font font, char c, int x, int y, int rx,
			int ry, float angle) {
		throw new RuntimeException();
	}

	/**
	 * Draws the character applying the given rotation and opacity level. This method uses the
	 * <code>nearest neighbor</code> algorithm to render the content. This algorithm is faster than
	 * <code>bilinear</code> algorithm but its rendering is more simple.
	 * <p>
	 * This method only supports characters in the range U+0000 to U+FFFF. It does not handle characters with code
	 * points greater than U+FFFF, which are represented as a pair of char values (referred to as "surrogate pair").
	 * <p>
	 * Some rotation samples are described in method comment
	 * {@link #drawRotatedImageBilinear(GraphicsContext, Image, int, int, int, int, float)}.
	 * <p>
	 * In addition with {@link #drawRotatedCharNearestNeighbor(GraphicsContext, Font, char, int, int, int, int, float)},
	 * this method allows to specify the global opacity value to apply during the image rendering.
	 *
	 * @param gc
	 *            the graphics context where render the drawing.
	 * @param font
	 *            the font to use.
	 * @param c
	 *            the character to draw.
	 * @param x
	 *            the x coordinate of the character reference anchor top-left point
	 * @param y
	 *            the y coordinate of the character reference anchor top-left point
	 * @param rx
	 *            the rotation center X coordinate.
	 * @param ry
	 *            the rotation center Y coordinate.
	 * @param angle
	 *            the rotation angle.
	 * @param alpha
	 *            the global opacity rendering value.
	 * @throws IllegalArgumentException
	 *             if the given alpha is not a value between {@link GraphicsContext#TRANSPARENT} and
	 *             {@link GraphicsContext#OPAQUE}.
	 * @see #drawRotatedCharBilinear(GraphicsContext, Font, char, int, int, int, int, float, int)
	 */
	public static void drawRotatedCharNearestNeighbor(GraphicsContext gc, Font font, char c, int x, int y, int rx,
			int ry, float angle, int alpha) {
		throw new RuntimeException();
	}

	/**
	 * Draws the image in the graphics context at given anchor top-left position and using the given scaling factor.
	 * This method uses the <code>bilinear</code> algorithm to render the image. This algorithm performs better
	 * rendering than <code>nearest neighbor</code> algorithm but it is slower to apply.
	 * <p>
	 * Equivalent to calling {@link #drawScaledImageBilinear(GraphicsContext, Image, int, int, float, float, int)} with
	 * {@link GraphicsContext#OPAQUE} as alpha.
	 *
	 * @param gc
	 *            the graphics context where to render the drawing.
	 * @param image
	 *            the image to draw
	 * @param x
	 *            the x coordinate of the image reference anchor top-left point
	 * @param y
	 *            the y coordinate of the image reference anchor top-left point
	 * @param factorX
	 *            the scaling X factor.
	 * @param factorY
	 *            the scaling Y factor.
	 * @throws IllegalArgumentException
	 *             if a factor is lower than to zero.
	 */
	public static void drawScaledImageBilinear(GraphicsContext gc, Image image, int x, int y, float factorX,
			float factorY) {
		throw new RuntimeException();
	}

	/**
	 * Draws the image in the graphics context at given anchor top-left position and using the given scaling factor and
	 * opacity level. This method uses the <code>bilinear</code> algorithm to render the image. This algorithm performs
	 * better rendering than <code>nearest neighbor</code> algorithm but it is slower to apply.
	 * <p>
	 * In addition with {@link #drawScaledImageBilinear(GraphicsContext, Image, int, int, float, float)}, this method
	 * allows to specify the global opacity value to apply during the image rendering.
	 *
	 * @param gc
	 *            the graphics context where to render the drawing.
	 * @param image
	 *            the image to draw
	 * @param x
	 *            the x coordinate of the image reference anchor top-left point
	 * @param y
	 *            the y coordinate of the image reference anchor top-left point
	 * @param factorX
	 *            the scaling X factor.
	 * @param factorY
	 *            the scaling Y factor.
	 * @param alpha
	 *            the global opacity rendering value.
	 * @throws IllegalArgumentException
	 *             if the given alpha is not a value between {@link GraphicsContext#TRANSPARENT} and
	 *             {@link GraphicsContext#OPAQUE} or if a factor is lower than to zero.
	 */
	public static void drawScaledImageBilinear(GraphicsContext gc, Image image, int x, int y, float factorX,
			float factorY, int alpha) {
		throw new RuntimeException();
	}

	/**
	 * Draws the image in the graphics context at given anchor top-left position and using the given scaling factor.
	 * This method uses the <code>nearest neighbor</code> algorithm to render the image. This algorithm is faster than
	 * <code>bilinear</code> algorithm but its rendering is more simple.
	 * <p>
	 * Equivalent to calling
	 * {@link #drawScaledImageNearestNeighbor(GraphicsContext, Image, int, int, float, float, int)} with
	 * {@link GraphicsContext#OPAQUE} as alpha.
	 *
	 * @param gc
	 *            the graphics context where to render the drawing.
	 * @param image
	 *            the image to draw
	 * @param x
	 *            the x coordinate of the image reference anchor top-left point
	 * @param y
	 *            the y coordinate of the image reference anchor top-left point
	 * @param factorX
	 *            the scaling X factor.
	 * @param factorY
	 *            the scaling Y factor.
	 * @throws IllegalArgumentException
	 *             if a factor is lower than to zero.
	 */
	public static void drawScaledImageNearestNeighbor(GraphicsContext gc, Image image, int x, int y, float factorX,
			float factorY) {
		throw new RuntimeException();
	}

	/**
	 * Draws the image in the graphics context at given anchor top-left position and using the given scaling factor and
	 * opacity level. This method uses the <code>nearest neighbor</code> algorithm to render the image. This algorithm
	 * is faster than <code>bilinear</code> algorithm but its rendering is more simple.
	 * <p>
	 * In addition with {@link #drawScaledImageNearestNeighbor(GraphicsContext, Image, int, int, float, float)}, this
	 * method allows to specify the global opacity value to apply during the image rendering.
	 *
	 * @param gc
	 *            the graphics context where to render the drawing.
	 * @param image
	 *            the image to draw
	 * @param x
	 *            the x coordinate of the image reference anchor top-left point
	 * @param y
	 *            the y coordinate of the image reference anchor top-left point
	 * @param factorX
	 *            the scaling X factor.
	 * @param factorY
	 *            the scaling Y factor.
	 * @param alpha
	 *            the global opacity rendering value.
	 * @throws IllegalArgumentException
	 *             if the given alpha is not a value between {@link GraphicsContext#TRANSPARENT} and
	 *             {@link GraphicsContext#OPAQUE} or if a factor is lower than to zero.
	 */
	public static void drawScaledImageNearestNeighbor(GraphicsContext gc, Image image, int x, int y, float factorX,
			float factorY, int alpha) {
		throw new RuntimeException();
	}

	/**
	 * Draws the character in the graphics context at given anchor top-left position and using the given scaling factor.
	 * This method uses the <code>bilinear</code> algorithm to render the character.
	 * <p>
	 * This method only supports characters in the range U+0000 to U+FFFF. It does not handle characters with code
	 * points greater than U+FFFF, which are represented as a pair of char values (referred to as "surrogate pair").
	 *
	 * @param gc
	 *            the graphics context where to render the drawing.
	 * @param character
	 *            the character to draw
	 * @param font
	 *            the font to use.
	 * @param x
	 *            the x coordinate of the character reference anchor top-left point
	 * @param y
	 *            the y coordinate of the character reference anchor top-left point
	 * @param factorX
	 *            the scaling X factor.
	 * @param factorY
	 *            the scaling Y factor.
	 * @throws IllegalArgumentException
	 *             if a factor is lower than to zero.
	 */
	public static void drawScaledCharBilinear(GraphicsContext gc, char character, Font font, int x, int y,
			float factorX, float factorY) {
		throw new RuntimeException();
	}

	/**
	 * Draws the string in the graphics context at given anchor top-left position and using the given scaling factor.
	 * This method uses the <code>bilinear</code> algorithm to render the string.
	 *
	 * @param gc
	 *            the graphics context where to render the drawing.
	 * @param string
	 *            the string to draw
	 * @param font
	 *            the font to use.
	 * @param x
	 *            the x coordinate of the character reference anchor top-left point
	 * @param y
	 *            the y coordinate of the character reference anchor top-left point
	 * @param factorX
	 *            the scaling X factor.
	 * @param factorY
	 *            the scaling Y factor.
	 * @throws IllegalArgumentException
	 *             if a factor is lower than to zero.
	 */
	public static void drawScaledStringBilinear(GraphicsContext gc, String string, Font font, int x, int y,
			float factorX, float factorY) {
		throw new RuntimeException();
	}

	/**
	 * Draws the substring in the graphics context at given anchor top-left position and using the given scaling factor.
	 * This method uses the <code>bilinear</code> algorithm to render the string.
	 *
	 * @param gc
	 *            the graphics context where to render the drawing.
	 * @param string
	 *            the string to draw
	 * @param offset
	 *            the index of the first character of the string.
	 * @param length
	 *            the number of characters of the string.
	 * @param font
	 *            the font to use.
	 * @param x
	 *            the x coordinate of the character reference anchor top-left point
	 * @param y
	 *            the y coordinate of the character reference anchor top-left point
	 * @param factorX
	 *            the scaling X factor.
	 * @param factorY
	 *            the scaling Y factor.
	 * @throws IllegalArgumentException
	 *             if a factor is lower than to zero.
	 * @throws StringIndexOutOfBoundsException
	 *             if the given offset and length do not specify a valid range within the given string.
	 */
	public static void drawScaledSubstringBilinear(GraphicsContext gc, String string, int offset, int length, Font font,
			int x, int y, float factorX, float factorY) {
		throw new RuntimeException();
	}

	/**
	 * Draws the renderable string in the graphics context at given anchor top-left position and using the given scaling
	 * factor. This method uses the <code>bilinear</code> algorithm to render the renderable string.
	 *
	 * @param gc
	 *            the graphics context where to render the drawing.
	 * @param string
	 *            the renderable string to draw
	 * @param x
	 *            the x coordinate of the character reference anchor top-left point
	 * @param y
	 *            the y coordinate of the character reference anchor top-left point
	 * @param factorX
	 *            the scaling X factor.
	 * @param factorY
	 *            the scaling Y factor.
	 * @throws IllegalArgumentException
	 *             if a factor is lower than to zero.
	 */
	public static void drawScaledRenderableStringBilinear(GraphicsContext gc, RenderableString string, int x, int y,
			float factorX, float factorY) {
		throw new RuntimeException();
	}

}
