/*
 * Java
 *
 * Copyright 2015-2023 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package ej.util;

import ej.bon.Constants;
import ej.sni.SNI;

/**
 * This class provides utility methods related to the device.
 */

public class Device {

	private static final String BUFFER_SIZE_CONSTANT_NAME = "ej.device.buffer.size";
	private static final int BUFFER_SIZE = Constants.getInt(BUFFER_SIZE_CONSTANT_NAME);

	/**
	 * Private constructor. This class is not intended to be instantiated as it only
	 * offers static methods.
	 */
	private Device() {
		// private constructor
	}

	/**
	 * Returns the name of the architecture of the device.
	 *
	 * @return the name of the architecture of the device.
	 */
	public static String getArchitecture() {
		byte[] buffer = new byte[BUFFER_SIZE];
		if (!DeviceNatives.getArchitecture(buffer, buffer.length)) {
			throw new IllegalArgumentException(); // wrong implementation
		}

		return SNI.toJavaString(buffer);
	}

	/**
	 * Returns the ID of the device. This identifier is guaranteed to be unique to
	 * all devices with the same architecture.
	 *
	 * @return the ID of the device.
	 */
	public static byte[] getId() {
		byte[] buffer = new byte[BUFFER_SIZE];
		int idSize = DeviceNatives.getId(buffer, buffer.length);
		if (idSize <= 0) {
			throw new IllegalArgumentException(); // wrong implementation
		}

		byte[] id = new byte[idSize];
		System.arraycopy(buffer, 0, id, 0, idSize);
		return id;
	}

	/**
	 * Returns the Version of the device. This can be a combination of hardware
	 * version(s) and version(s) of software already installed on the Device.
	 *
	 * @return the Version of the device.
	 */
	public static String getVersion() {
		byte[] buffer = new byte[BUFFER_SIZE];
		DeviceNatives.getVersion(buffer);
		return SNI.toJavaString(buffer);
	}

	/**
	 * Reboots the device.
	 */
	public static void reboot() {
		DeviceNatives.reboot();
	}

	/**
	 * Shutdowns the device.
	 */
	public static void shutdown() {
		DeviceNatives.shutdown();
	}

}
