/*
 * Java
 *
 * Copyright 2020-2025 MicroEJ Corp. All rights reserved.
 * MicroEJ Corp. PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package ej.microui.display;

import java.nio.ByteOrder;
import java.util.ServiceLoader;

/**
 * This interface represents the MicroUI Graphics Engine (MicroUI framework). It provides some methods to map MicroUI
 * byte arrays in MicroUI Graphics Context objects, to manipulate MicroUI colors etc.
 * <p>
 * An instance of this framework is available via the {@link #Instance} field. To use the framework methods, caller must
 * synchronize its calls on this {@link #Instance} field.
 */
public interface LLUIDisplay {

	/**
	 * Unique instance of MicroUI Graphics Engine. It provides some methods to synchronize drawings between them, to map
	 * MicroUI byte arrays in graphical objects, to manipulate MicroUI colors etc.
	 * <p>
	 * To use the framework methods, caller must synchronize its calls on this {@link #Instance} field.
	 */
	LLUIDisplay Instance = ServiceLoader.load(LLUIDisplay.class, LLUIDisplay.class.getClassLoader()).iterator().next();

	/**
	 * Unique instance of MicroUI Graphics Engine's internal software drawer. It is an instance of {@link UIDrawing} and
	 * it implements all drawings in software. The rendering is exactly the same than the embedded algorithms listed in
	 * <code>ui_drawing_soft.h</code>.
	 * <p>
	 * This drawer converts the MicroUI graphics context color, respects the clip and updates the new
	 * <code>Display.flush()</code> region (dirty region) as mentioned in {@link LLUIPainter} class comment.
	 *
	 * @return an implementation of {@link UIDrawing} in software.
	 */
	UIDrawing getUIDrawerSoftware();

	/**
	 * Maps a MicroUI GraphicsContext byte array (retrieved by MicroEJ application calling
	 * <code>gc.getSNIContext()</code>) on a {@link MicroUIGraphicsContext}.
	 * <p>
	 * Implementation notes:
	 * <ul>
	 * <li>Same object is always returned, only internal {@link MicroUIGraphicsContext} fields are updated. By
	 * consequence this method can only be called once during a MicroUI native call. See
	 * {@link #newMicroUIGraphicsContext(byte[])} to force to create a new {@link MicroUIGraphicsContext} instead.</li>
	 * <li>The implementation assumes given byte array has been retrieved by MicroEJ application calling
	 * <code>gc.getSNIContext()</code>. If given byte array does not respect this rule, the behavior is unknown.</li>
	 * </ul>
	 *
	 * @param gc
	 *            the MicroUI GraphicsContext representation.
	 * @return the unique {@link MicroUIGraphicsContext} object updated with the given MicroUI GraphicsContext
	 *         characteristics.
	 */
	MicroUIGraphicsContext mapMicroUIGraphicsContext(byte[] gc);

	/**
	 * Creates a new {@link MicroUIGraphicsContext} for the given MicroUI GraphicsContext byte array. This allows to
	 * manipulate several destination targets in same native call. However it takes more time to create a new instance
	 * at each native call than mapping the unique instance.
	 * <p>
	 * The implementation assumes given byte array has been retrieved by MicroEJ application calling
	 * <code>gc.getSNIContext()</code>. If given byte array does not respect this rule, the behavior is unknown.
	 *
	 * @param gc
	 *            the MicroUI GraphicsContext representation.
	 * @return a new {@link MicroUIGraphicsContext} object updated with the given MicroUI GraphicsContext
	 *         characteristics.
	 *
	 * @see LLUIDisplay#mapMicroUIGraphicsContext(byte[])
	 */
	MicroUIGraphicsContext newMicroUIGraphicsContext(byte[] gc);

	/**
	 * Maps a MicroUI Image byte array (retrieved by MicroEJ application calling <code>image.getSNIContext()</code>) on
	 * a {@link MicroUIImage}. This image can be used as source (copy image in a graphics context) or as target (copy
	 * image in another image).
	 * <p>
	 * Implementation notes:
	 * <ul>
	 * <li>Same object is always returned, only internal {@link MicroUIImage} fields are updated. By consequence this
	 * method can only be called once during a MicroUI native call. See {@link #newMicroUIImage(byte[])} to force to
	 * create a new {@link MicroUIImage} instead.</li>
	 * <li>The implementation assumes given byte array has been retrieved by MicroEJ application calling
	 * <code>image.getSNIContext()</code>. If given byte array does not respect this rule, the behavior is unknown.</li>
	 * </ul>
	 *
	 * @param image
	 *            the MicroUI Image representation.
	 * @return the unique {@link MicroUIImage} object updated with the given MicroUI Image characteristics.
	 */
	MicroUIImage mapMicroUIImage(byte[] image);

	/**
	 * Creates a new {@link MicroUIImage} for the given MicroUI Image byte array. This allows to manipulate several
	 * sources and targets in same native call. However it takes more time to create a new instance at each native call
	 * than mapping the unique instance.
	 * <p>
	 * The implementation assumes given byte array has been retrieved by MicroEJ application calling
	 * <code>image.getSNIContext()</code>. If given byte array does not respect this rule, the behavior is unknown.
	 *
	 * @param image
	 *            the MicroUI Image representation.
	 * @return a new {@link MicroUIImage} object updated with the given MicroUI Image characteristics.
	 *
	 * @see LLUIDisplay#mapMicroUIImage(byte[])
	 */
	MicroUIImage newMicroUIImage(byte[] image);

	/**
	 * Maps a MicroUI Font byte array (retrieved by MicroEJ application calling <code>font.getSNIContext()</code>) on a
	 * {@link MicroUIFont}.
	 * <p>
	 * Implementation notes:
	 * <ul>
	 * <li>Same object is always returned, only internal {@link MicroUIFont} fields are updated. By consequence this
	 * method can only be called once during a MicroUI native call.</li>
	 * <li>The implementation assumes given byte array has been retrieved by MicroEJ application calling
	 * <code>font.getSNIContext()</code>. If given byte array does not respect this rule, the behavior is unknown.</li>
	 * </ul>
	 *
	 * @param font
	 *            the MicroUI Font representation.
	 * @return the unique {@link MicroUIFont} object updated with the given MicroUI Font characteristics.
	 */
	MicroUIFont mapMicroUIFont(byte[] font);

	/**
	 * Maps a MicroUI RenderableString byte array (retrieved by MicroEJ application calling
	 * <code>renderableString.getSNIContext()</code>) on a {@link MicroUIRenderableString}.
	 * <p>
	 * Implementation notes:
	 * <ul>
	 * <li>Same object is always returned, only internal {@link MicroUIRenderableString} fields are updated. By
	 * consequence this method can only be called once during a MicroUI native call.</li>
	 * <li>The implementation assumes given byte array has been retrieved by MicroEJ application calling
	 * <code>renderableString.getSNIContext()</code>. If given byte array does not respect this rule, the behavior is
	 * unknown.</li>
	 * </ul>
	 *
	 * @param renderableString
	 *            the MicroUI RenderableString representation
	 * @return the unique {@link MicroUIRenderableString} object updated with the given MicroUI RenderableString
	 *         characteristics
	 */
	MicroUIRenderableString mapMicroUIRenderableString(byte[] renderableString);

	/**
	 * Maps a char array subset on a string.
	 *
	 * @param chars
	 *            the char array
	 * @param offset
	 *            the offset in the char array
	 * @param length
	 *            the length
	 * @return a char array representing the string
	 */
	char[] mapString(char[] chars, int offset, int length);

	/**
	 * Gets the number of bits (opacity included) to represent a pixel on the hardware device. For instance a RGB565
	 * display returns 16.
	 *
	 * @return the hardware device pixel depth.
	 */
	int getDisplayPixelDepth();

	/**
	 * Requests a call to {@link LLUIDisplayImpl#flush()} if something has been drawn in the back buffer (application
	 * dirty region is not null). The call of {@link LLUIDisplayImpl#flush()} is synchronized with the MicroEJ
	 * application drawing (see MicroUI event pump).
	 */
	void requestFlush();

	/**
	 * Requests a call to <code>Displayable.render()</code> (current displayable on display). The call of
	 * <code>Displayable.render()</code> is synchronized with the MicroEJ application drawing (see MicroUI event pump).
	 */
	void requestRender();

	/**
	 * Converts the given MicroUI color (format 0xAARRGGBB) in a truncated color (same format: 0xAARRGGBB).
	 * <p>
	 * This color takes in consideration the hardware display pixel representation constraints. For instance, for a
	 * display with the pixel format RGB565, the truncated color of green color (0xff00ff00) is 0xff00fc00 (because
	 * RGB565 format keeps only 6 first bits for green component).
	 * <p>
	 * This truncated color must be used to convert the MicroUI GraphicsContext current color in order to respect the
	 * hardware display pixel representation constraints.
	 *
	 * @param color
	 *            the MicroUI color in format 0xAARRGGBB.
	 * @return the truncated color in format 0xAARRGGBB.
	 */
	int convertARGBColorToColorToDraw(int color);

	/**
	 * Converts the colors of the pixels in the given image's region.
	 *
	 * This is useful when the given region has been drawn by a third-party algorithm that doesn't respect the image
	 * format restrictions. For instance an anti-aliased drawing can draw a pixel with the color 0xfff3f3f3. On a RGB565
	 * image, this color must be converted in 0xfff0f0f0.
	 *
	 * The implementation consists to call {@link LLUIDisplay#convertARGBColorToColorToDraw(int)} for each region's
	 * pixel.
	 *
	 * @param image
	 *            the image to update.
	 * @param x
	 *            the top-left pixel X coordinate.
	 * @param y
	 *            the top-left pixel Y coordinate.
	 * @param width
	 *            the region width.
	 * @param height
	 *            the region height.
	 */
	void convertRegion(MicroUIImage image, int x, int y, int width, int height);

	/**
	 * Blends two colors applying a global alpha factor.
	 *
	 * @param foreground
	 *            the ARGB8888 foreground color.
	 * @param background
	 *            the ARGB8888 background color.
	 * @param alpha
	 *            the global alpha factor.
	 *
	 * @return an ARGB8888 color.
	 */
	int blend(int foreground, int background, int alpha);

	/**
	 * Gets the display format.
	 *
	 * @see MicroUIImageFormat
	 *
	 * @return the display format.
	 */
	MicroUIImageFormat getDisplayFormat();

	/**
	 * Tells if format is the display format or not.
	 *
	 * @see MicroUIImageFormat
	 *
	 * @param format
	 *            the format to check.
	 * @return true if format refers to the display format, false otherwise
	 */
	boolean isDisplayFormat(MicroUIImageFormat format);

	/**
	 * Tells if format is a custom format or not.
	 *
	 * @see MicroUIImageFormat
	 *
	 * @param format
	 *            the format to check.
	 * @return true if format refers to a custom format, false otherwise
	 */
	boolean isCustomFormat(MicroUIImageFormat format);

	/**
	 * Registers a drawer that supports a specific format.
	 *
	 * @param drawer
	 *            the drawer
	 * @see UIDrawing#handledFormat()
	 */
	void registerUIDrawer(UIDrawing drawer);

	/**
	 * Unregisters the drawer registered for the given format.
	 *
	 * @param format
	 *            the now unsupported format
	 * @see UIDrawing#handledFormat()
	 */
	void unregisterUIDrawer(MicroUIImageFormat format);

	/**
	 * Gets the drawer used to draw on the given graphics context.
	 *
	 * @param gc
	 *            the graphics context to draw on
	 * @return the drawer registered for the format of the graphics context or a stub if none
	 */
	UIDrawing getUIDrawer(MicroUIGraphicsContext gc);

	/**
	 * Gets the drawer used to draw with the given format.
	 *
	 * @param format
	 *            the format
	 * @return the drawer registered for the format or a stub if none
	 */
	UIDrawing getUIDrawer(MicroUIImageFormat format);

	/**
	 * Registers a drawer that supports a specific format.
	 *
	 * @param drawer
	 *            the drawer
	 * @see UIImageDrawing#handledFormat()
	 */
	void registerUIImageDrawer(UIImageDrawing drawer);

	/**
	 * Unregisters the drawer registered for the given format.
	 *
	 * @param format
	 *            the now unsupported format
	 * @see UIImageDrawing#handledFormat()
	 */
	void unregisterUIImageDrawer(MicroUIImageFormat format);

	/**
	 * Gets the image drawer used to draw the given image.
	 *
	 * @param image
	 *            the image to draw
	 * @return the current image drawer (the default drawer if no other drawer has been set)
	 */
	UIImageDrawing getUIImageDrawer(MicroUIImage image);

	/**
	 * Gets the image drawer used to draw the images with the given format.
	 *
	 * @param format
	 *            the format of the image to draw
	 * @return the current image drawer (the default drawer if no other drawer has been set)
	 */
	UIImageDrawing getUIImageDrawer(MicroUIImageFormat format);

	/**
	 * Registers a buffered image provider.
	 * <p>
	 * It will be used to create buffered image for the format it handles
	 * ({@link BufferedImageProvider#newBufferedImage(int, int)}).
	 * <p>
	 * It will also be registered as is also registered as drawer {@link #registerUIDrawer(UIDrawing)} and custom image
	 * drawer {@link #registerUIImageDrawer(UIImageDrawing)}.
	 *
	 * @param provider
	 *            the buffered image provider
	 */
	void registerBufferedImageProvider(BufferedImageProvider provider);

	/**
	 * Unregisters the buffered image provider registered for a specific format.
	 *
	 * @param format
	 *            the format
	 */
	void unregisterBufferedImageProvider(MicroUIImageFormat format);

	/**
	 * Gets the buffered image provider used to create images with the given format.
	 *
	 * @param format
	 *            the format of the created images
	 * @return the current buffered image provider (or {@code null} if none)
	 */
	BufferedImageProvider getBufferedImageProvider(MicroUIImageFormat format);

	/**
	 * Tells if format is a custom format or not.
	 *
	 * @see MicroUIFontFormat
	 *
	 * @param format
	 *            the format to check.
	 * @return true if format refers to a custom format, false otherwise
	 */
	boolean isCustomFormat(MicroUIFontFormat format);

	/**
	 * Gets the font drawer used to draw the given font.
	 *
	 * @param font
	 *            the font to draw
	 * @return the current font drawer (the default drawer if no other drawer has been set)
	 */
	UIFontDrawing getUIFontDrawer(MicroUIFont font);

	/**
	 * Gets the font drawer used to draw the fonts with the given format.
	 *
	 * @param format
	 *            the format of the font to draw
	 * @return the current font drawer (the default drawer if no other drawer has been set)
	 */
	UIFontDrawing getUIFontDrawer(MicroUIFontFormat format);

	/**
	 * Registers a drawer that supports a specific format.
	 *
	 * @param drawer
	 *            the drawer
	 * @see UIFontDrawing#handledFormat()
	 */
	void registerUIFontDrawer(UIFontDrawing drawer);

	/**
	 * Unregisters the drawer registered for the given format.
	 *
	 * @param format
	 *            the now unsupported format
	 * @see UIFontDrawing#handledFormat()
	 */
	void unregisterUIFontDrawer(MicroUIFontFormat format);

	/**
	 * Returns an image that targets the front buffer (display buffer) instead of the back buffer.
	 * <p>
	 * This behavior only concerns the following use case:
	 * <ul>
	 * <li>the image targets the display (not an image),</li>
	 * <li>the display buffer refresh strategy (BRS) has not restored yet the content of the old back buffer to the
	 * current back buffer.</li>
	 * </ul>
	 * In that case, the read actions (GraphicsContext.readPixel(), Painter.drawDisplayRegion(), etc.) cannot use the
	 * back buffer as source buffer. The algorithm has to call this method to retrieve the front buffer.
	 * <p>
	 * For all other use cases, the returned image is the given parameter.
	 *
	 * @param image
	 *            the Java image.
	 * @return the result image.
	 */
	MicroUIImage getSource(MicroUIImage image);

	/**
	 * Gets the endianness of the architecture.
	 *
	 * @return the endianness
	 */
	ByteOrder getEndianness();

}
