/*
 * Java
 *
 * Copyright 2025 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package com.microej.kf.util.policy;

import com.microej.kf.util.control.fs.FileSystemResourcesController;
import com.microej.kf.util.control.net.Bandwidth;
import com.microej.kf.util.control.net.NetResourcesController;
import com.microej.kf.util.control.net.Subnet;
import com.microej.kf.util.module.SandboxedModule;
import com.microej.kf.util.module.SandboxedModuleHelper;
import ej.kf.Feature;
import ej.kf.Kernel;
import ej.kf.Module;
import ej.service.ServiceFactory;
import org.json.me.JSONArray;
import org.json.me.JSONException;
import org.json.me.JSONObject;
import org.json.me.JSONTokener;

import java.io.IOException;
import java.io.InputStream;
import java.util.Set;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * This class enforces a policy on a feature that defines resource limitations (CPU, RAM, flash, network bandwidth),
 * priority and criticality.
 */
public class FeaturePolicyEnforcer {
	private int threadPriority;
	private static final String POLICY_PROPERTY = "feature.policy.name"; // $NON-NLS-1$
	private static final String POLICY_FILE_NAME = "feature.policy.json"; // $NON-NLS-1$
	private static final int QUOTA_PERCENTAGE_RATIO = Integer.getInteger("monitoring.check.cpu.calibration", 100_000);
	private static final Logger LOGGER = Logger.getLogger(FeaturePolicyEnforcer.class.getName());

	/**
	 * Initializes a policy enforcer.
	 */
	public FeaturePolicyEnforcer() {
		this.threadPriority = Thread.NORM_PRIORITY;
	}

	/**
	 * Applies a policy to the given feature.
	 *
	 * @param app
	 * 		the feature to which the policy will be applied.
	 * @throws JSONException
	 * 		if file parsing fails.
	 * @throws IOException
	 * 		if I/O errors occur.
	 */
	public void applyPolicy(Feature app) throws JSONException, IOException {
		if (System.getProperty("monitoring.check.cpu.calibration") == null && LOGGER.isLoggable(Level.WARNING)) {
			LOGGER.log(Level.WARNING,
					"Property monitoring.check.cpu.calibration is not set. Running with default value: "
							+ QUOTA_PERCENTAGE_RATIO);
		}

		if (LOGGER.isLoggable(Level.INFO)) {
			LOGGER.log(Level.INFO, "Loading policy file for app " + app.getName());
		}

		String resourceFileName = System.getProperty(POLICY_PROPERTY, POLICY_FILE_NAME);
		String resourceFilePath = resourceFileName.startsWith("/") ? resourceFileName : "/" + resourceFileName;

		try (InputStream inputStream = app.getResourceAsStream(resourceFilePath)) {
			if (inputStream == null) {
				if (LOGGER.isLoggable(Level.FINE)) {
					LOGGER.log(Level.FINE, "Policy file was not found at: " + resourceFilePath);
				}
				return;
			}
			loadResourceFile(inputStream, app);
		}
	}

	private void loadResourceFile(InputStream inputStream, Feature app) throws JSONException, IOException {
		JSONTokener jsonTokener = new JSONTokener(inputStream);
		JSONObject rootObject = new JSONObject(jsonTokener);

		// Threads priority between [1 - 10].
		this.threadPriority = rootObject.optInt(FeaturePolicySchema.PRIORITY.toString(), Thread.NORM_PRIORITY);
		// Feature criticality between [1 - 10].
		int criticality = rootObject.optInt(FeaturePolicySchema.CRITICALITY.toString(), Feature.NORM_CRITICALITY);

		int cpuLimit = -1;
		int ramLimit = -1;
		int flashLimit = -1;

		if (rootObject.has(FeaturePolicySchema.RESOURCES.toString()) && rootObject.get(
				FeaturePolicySchema.RESOURCES.toString()) instanceof JSONObject) {
			JSONObject resources = rootObject.getJSONObject(FeaturePolicySchema.RESOURCES.toString());
			if (resources.has(FeaturePolicySchema.LIMIT.toString()) && resources.get(
					FeaturePolicySchema.LIMIT.toString()) instanceof JSONObject) {
				JSONObject resourcesLimits = resources.getJSONObject(FeaturePolicySchema.LIMIT.toString());

				// CPU limit in %.
				cpuLimit = resourcesLimits.optInt(FeaturePolicySchema.CPU.toString(), -1);
				// RAM limit in bytes.
				ramLimit = resourcesLimits.optInt(FeaturePolicySchema.RAM.toString(), -1);
				// Flash limit in bytes.
				flashLimit = resourcesLimits.optInt(FeaturePolicySchema.FLASH.toString(), -1);
			}
		}

		if (rootObject.has(FeaturePolicySchema.NET_DATA.toString())) {
			JSONArray networkData = rootObject.getJSONObject(FeaturePolicySchema.NET_DATA.toString())
					.optJSONArray(FeaturePolicySchema.NET_UPSTREAM.toString());
			if (networkData == null) {
				networkData = new JSONArray();
				networkData.put(rootObject.getJSONObject(FeaturePolicySchema.NET_DATA.toString())
						.getJSONObject(FeaturePolicySchema.NET_UPSTREAM.toString()));
			}

			for (int i = 0; i < networkData.length(); i++) {
				JSONObject networkResources = networkData.getJSONObject(i);
				// Network bandwidth in bytes per day.
				int networkBandwidthLimit = networkResources.optInt(FeaturePolicySchema.NET_LIMIT.toString(), -1);

				Bandwidth bandwidth = new Bandwidth(Bandwidth.Period.DAY, -1, networkBandwidthLimit);
				// Subnet CIDR.
				String cidrSubnet = networkResources.optString(FeaturePolicySchema.NET_SUBNET.toString(), "0.0.0.0/0");
				Subnet subnet = Subnet.fromCidrNotation(cidrSubnet);

				applyNetworkBandwidthLimit(app, subnet, bandwidth);

				if (LOGGER.isLoggable(Level.INFO)) {
					LOGGER.log(Level.INFO, "[Network " + i + "] Bandwidth: " + bandwidth);
					LOGGER.log(Level.INFO, "[Network " + i + "] Subnet: " + subnet);
				}
			}
		}

		if (LOGGER.isLoggable(Level.INFO)) {
			LOGGER.log(Level.INFO, "Threads priority: " + this.threadPriority);
			LOGGER.log(Level.INFO, "Feature criticality: " + criticality);
			LOGGER.log(Level.INFO, "CPU limit: " + cpuLimit);
			LOGGER.log(Level.INFO, "RAM limit: " + ramLimit);
			LOGGER.log(Level.INFO, "Flash limit: " + flashLimit);
		}

		applyCriticality(app, criticality);
		applyCpuLimits(app, cpuLimit);
		applyRamLimit(app, ramLimit);
		applyFlashLimit(app, flashLimit);
	}

	/**
	 * Sets priority to the threads owned by the given feature.
	 *
	 * @param app
	 * 		the feature to which the priority will be applied.
	 */
	public void applyThreadPriority(Feature app) {
		final int priority = this.threadPriority;

		if (priority == Thread.NORM_PRIORITY) {
			// Priority is set to default, nothing to do.
			return;
		}

		if (priority < Thread.MIN_PRIORITY || priority > Thread.MAX_PRIORITY) {
			if (LOGGER.isLoggable(Level.SEVERE)) {
				LOGGER.log(Level.SEVERE,
						"Invalid priority: " + priority + ". Expected between " + Thread.MIN_PRIORITY + " and "
								+ Thread.MAX_PRIORITY + ". Running with normal priority");
			}
			return;
		}

		Set<Thread> threads = Thread.getAllStackTraces().keySet();
		for (Thread thread : threads) {
			Module owner = Kernel.getOwner(thread);
			if (owner.equals(app)) {
				thread.setPriority(priority);
				if (LOGGER.isLoggable(Level.INFO)) {
					LOGGER.log(Level.INFO,
							"Thread name = " + thread.getName() + ", owner = " + owner.getName() + ", priority = "
									+ thread.getPriority());
				}
			}
		}

		if (LOGGER.isLoggable(Level.INFO)) {
			LOGGER.log(Level.INFO, "Threads priority applied: " + priority);
		}
	}

	/**
	 * Sets the criticality of the given feature.
	 *
	 * @param app
	 * 		the eature to which the criticality will be applied.
	 * @param criticality
	 * 		the criticality level to apply.
	 */
	private void applyCriticality(Feature app, int criticality) {
		if (criticality == Feature.NORM_CRITICALITY) {
			// Criticality is set to default, nothing to do.
			return;
		}

		if (criticality < Feature.MIN_CRITICALITY || criticality > Feature.MAX_CRITICALITY) {
			if (LOGGER.isLoggable(Level.SEVERE)) {
				LOGGER.log(Level.SEVERE,
						"Invalid criticality: " + criticality + ". Expected between " + Feature.MIN_CRITICALITY
								+ " and " + Feature.MAX_CRITICALITY + ". Running with normal criticality");
			}
			return;
		}

		try {
			app.setCriticality(criticality);

			if (LOGGER.isLoggable(Level.INFO)) {
				LOGGER.log(Level.INFO, "Criticality applied: " + app.getCriticality());
			}
		} catch (Exception e) {
			if (LOGGER.isLoggable(Level.SEVERE)) {
				LOGGER.log(Level.SEVERE, "Error while applying criticality", e);
			}
		}
	}

	/**
	 * Sets the execution quota allocated to the threads owned by the given feature.
	 *
	 * @param app
	 * 		the feature to which the execution quota will be applied.
	 * @param cpuLimit
	 * 		the execution quota to apply.
	 */
	private void applyCpuLimits(Feature app, int cpuLimit) {
		try {
			int quota;
			if (cpuLimit == -1 || cpuLimit == 0) {
				quota = cpuLimit;
			} else {
				quota = (int) ((cpuLimit / 100.0) * QUOTA_PERCENTAGE_RATIO);
			}

			if (LOGGER.isLoggable(Level.INFO)) {
				LOGGER.log(Level.INFO, "CPU quota computed: " + quota);
			}
			app.setExecutionQuota(quota);

			if (LOGGER.isLoggable(Level.INFO)) {
				LOGGER.log(Level.INFO, "CPU quota applied: " + app.getExecutionQuota());
			}
		} catch (Exception e) {
			if (LOGGER.isLoggable(Level.SEVERE)) {
				LOGGER.log(Level.SEVERE, "Error while applying CPU limit", e);
			}
		}
	}

	/**
	 * Sets the maximum amount of memory heap that can be allocated by the given feature.
	 *
	 * @param app
	 * 		the feature to which the RAM limit will be applied.
	 * @param ramLimit
	 * 		the RAM limit to apply.
	 */
	private void applyRamLimit(Feature app, int ramLimit) {
		if (ramLimit == -1) {
			if (LOGGER.isLoggable(Level.INFO)) {
				LOGGER.log(Level.INFO, "Skipping RAM limit");
			}
			return;
		}

		// Check if ramLimit < Runtime.maxMemory()?

		try {
			app.setMemoryLimit(ramLimit);

			if (LOGGER.isLoggable(Level.INFO)) {
				LOGGER.log(Level.INFO, "RAM limit applied: " + app.getMemoryLimit());
			}
		} catch (Exception e) {
			if (LOGGER.isLoggable(Level.SEVERE)) {
				LOGGER.log(Level.SEVERE, "Error while applying RAM limit", e);
			}
		}
	}

	/**
	 * Sets the maximum amount of flash memory that can be allocated by the given feature.
	 *
	 * @param app
	 * 		the feature to which the flash limit will be applied.
	 * @param flashLimit
	 * 		the flash limit to apply.
	 */
	private void applyFlashLimit(Feature app, int flashLimit) {
		SandboxedModuleHelper moduleManager = ServiceFactory.getRequiredService(SandboxedModuleHelper.class);
		SandboxedModule sandboxedModule = moduleManager.getModule(app.getName());
		if (sandboxedModule != null) {
			FileSystemResourcesController fsController = sandboxedModule.getFileSystemResourceController();
			if (flashLimit > 0) {
				fsController.setMaxStorageSize(flashLimit);
			}
			if (LOGGER.isLoggable(Level.INFO)) {
				LOGGER.log(Level.INFO, "Flash limit applied: " + fsController.getMaxStorageSize());
			}
		}
	}

	/**
	 * Sets the maximum network bandwidth allowed for a given feature on a given subnet.
	 *
	 * @param app
	 * 		the feature to which the network bandwidth will be applied.
	 * @param subnet
	 * 		the subnet for which the bandwidth is set.
	 * @param maxBandwidth
	 * 		the maximum bandwidth to apply.
	 */
	private void applyNetworkBandwidthLimit(Feature app, Subnet subnet, Bandwidth maxBandwidth) {
		SandboxedModuleHelper moduleManager = ServiceFactory.getRequiredService(SandboxedModuleHelper.class);
		SandboxedModule sandboxedModule = moduleManager.getModule(app.getName());
		if (sandboxedModule != null) {
			NetResourcesController netController = sandboxedModule.getNetworkResourceController();
			netController.setMaxBandwidth(subnet, maxBandwidth);
			if (LOGGER.isLoggable(Level.INFO)) {
				LOGGER.log(Level.INFO, "Network bandwidth applied");
			}
		}
	}
}

