/*
 * Java
 *
 * Copyright 2021-2025 MicroEJ Corp.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package com.microej.microvg.test;

import static org.junit.Assert.assertEquals;

import org.junit.AfterClass;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;

import ej.microui.display.Colors;
import ej.microui.display.Display;
import ej.microui.display.GraphicsContext;
import ej.microvg.BlendMode;
import ej.microvg.BufferedVectorImage;
import ej.microvg.LinearGradient;
import ej.microvg.Matrix;
import ej.microvg.Path;
import ej.microvg.VectorGraphicsPainter;
import ej.microvg.VectorGraphicsPainter.FillType;
import ej.microvg.VectorImage;

/**
 * Tests the linear gradient.
 */
@SuppressWarnings("nls")
public class TestGradient {

	/**
	 * The black background (0x000000) prevents detection of empty drawings: use white instead.
	 */
	private static final int BACKGROUND_COLOR = Colors.WHITE;

	/**
	 * Starts MicroUI.
	 */
	@BeforeClass
	public static void pre() {
		TestUtilities.startMicroUI();
	}

	/**
	 * Stops MicroUI.
	 */
	@AfterClass
	public static void post() {
		TestUtilities.stopMicroUI();
	}

	/**
	 * Resets the content of the screen to white.
	 */
	@Before
	public static void preTest() {
		Display.getDisplay().getGraphicsContext().reset();
		TestUtilities.clearScreen(BACKGROUND_COLOR);
	}

	/**
	 * Tests an horizontal gradient.
	 */
	@Test
	public static void testGradientHorizontal() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		Path path = new Path();
		path.moveTo(0, 0);
		path.lineTo(100, 0);
		path.lineTo(100, 100);
		path.lineTo(0, 100);
		path.close();

		LinearGradient gradient = new LinearGradient(0, 0, 99, 0, new int[] { 0xffff0000, 0xffffff00, 0xffffffff });
		VectorGraphicsPainter.fillGradientPath(g, path, new Matrix(), gradient);
		display.flush();

		TestUtilities.check("left", 0, 0, g, Colors.RED);
		TestUtilities.check("center", 49, 0, g, Colors.YELLOW);
		TestUtilities.check("right", 99, 0, g, Colors.WHITE);
	}

	/**
	 * Tests an horizontal gradient with stops.
	 */
	@Test
	public static void testGradientHorizontalStops() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		Path path = new Path();
		path.moveTo(0, 0);
		path.lineTo(100, 0);
		path.lineTo(100, 100);
		path.lineTo(0, 100);
		path.close();

		LinearGradient gradient = new LinearGradient(0, 0, 99, 0, new int[] { 0xffff0000, 0xffffff00, 0xffffffff },
				new float[] { 0f, 0.25f, 1f });
		VectorGraphicsPainter.fillGradientPath(g, path, new Matrix(), gradient);
		display.flush();

		TestUtilities.check("left", 0, 0, g, Colors.RED);
		TestUtilities.check("center", 25, 0, g, Colors.YELLOW);
		TestUtilities.check("right", 99, 0, g, Colors.WHITE);
	}

	/**
	 * Tests a diagonal gradient.
	 */
	@Test
	public static void testGradientDiagonal() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		Path path = new Path();
		path.moveTo(0, 0);
		path.lineTo(100, 0);
		path.lineTo(100, 100);
		path.lineTo(0, 100);
		path.close();

		LinearGradient gradient = new LinearGradient(0, 0, 99, 99, new int[] { 0xffff0000, 0xffffff00, 0xffffffff },
				new float[] { 0.0f, 0.5f, 1.0f });
		VectorGraphicsPainter.fillGradientPath(g, path, new Matrix(), gradient);
		display.flush();

		TestUtilities.check("top left", 0, 0, g, Colors.RED);
		TestUtilities.check("bottom right", 99, 99, g, Colors.WHITE);
	}

	/**
	 * Tests a translated horizontal gradient.
	 */
	@Test
	public static void testGradientHorizontalTranslate() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		Path path = new Path();
		path.moveTo(50, 50);
		path.lineTo(150, 50);
		path.lineTo(150, 150);
		path.lineTo(50, 150);
		path.close();

		LinearGradient gradient = new LinearGradient(0, 0, 99, 0, new int[] { 0xffff0000, 0xffffff00, 0xffffffff },
				new float[] { 0.0f, 0.5f, 1.0f });
		Matrix gradientMatrix = gradient.getMatrix();
		gradientMatrix.setTranslate(50, 50);
		VectorGraphicsPainter.fillGradientPath(g, path, new Matrix(), gradient);
		display.flush();

		TestUtilities.check("left", 50, 50, g, Colors.RED);
		TestUtilities.check("center", 99, 50, g, Colors.YELLOW);
		TestUtilities.check("right", 149, 50, g, Colors.WHITE);
	}

	/**
	 * Tests an horizontal gradient in a translated path.
	 */
	@Test
	public static void testGradientHorizontalGlobalTranslate() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		Path path = new Path();
		path.moveTo(0, 0);
		path.lineTo(100, 0);
		path.lineTo(100, 100);
		path.lineTo(0, 100);
		path.close();

		LinearGradient gradient = new LinearGradient(0, 0, 99, 0, new int[] { 0xffff0000, 0xffffff00, 0xffffffff },
				new float[] { 0.0f, 0.5f, 1.0f });

		Matrix matrix = new Matrix();
		matrix.setTranslate(50, 50);

		VectorGraphicsPainter.fillGradientPath(g, path, matrix, gradient);
		display.flush();

		TestUtilities.check("left", 50, 50, g, Colors.RED);
		TestUtilities.check("center", 99, 50, g, Colors.YELLOW);
		TestUtilities.check("right", 149, 50, g, Colors.WHITE);
	}

	/**
	 * Tests a scaled horizontal gradient in a translated path.
	 */
	@Test
	public static void testGradientHorizontalScaleGlobalTranslate() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		Path path = new Path();
		path.moveTo(0, 0);
		path.lineTo(100, 0);
		path.lineTo(100, 100);
		path.lineTo(0, 100);
		path.close();

		LinearGradient gradient = new LinearGradient(0, 0, 49, 0, new int[] { 0xffff0000, 0xffffff00, 0xffffffff },
				new float[] { 0.0f, 0.5f, 1.0f });
		Matrix gradientMatrix = gradient.getMatrix();
		gradientMatrix.setScale(2, 1);

		Matrix matrix = new Matrix();
		matrix.setTranslate(50, 50);

		VectorGraphicsPainter.fillGradientPath(g, path, matrix, gradient);
		display.flush();

		TestUtilities.check("left", 50, 50, g, Colors.RED);
		TestUtilities.check("center", 99, 50, g, Colors.YELLOW);
		TestUtilities.check("right", 149, 50, g, Colors.WHITE);
	}

	/**
	 * Tests a translated horizontal gradient in a scaled path.
	 */
	@Test
	public static void testGradientHorizontalTranslateGlobalScale() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		Path path = new Path();
		path.moveTo(0, 0);
		path.lineTo(100, 0);
		path.lineTo(100, 100);
		path.lineTo(0, 100);
		path.close();

		LinearGradient gradient = new LinearGradient(50, 0, 149, 0, new int[] { 0xffff0000, 0xffffff00, 0xffffffff },
				new float[] { 0.0f, 0.5f, 1.0f });
		Matrix gradientMatrix = gradient.getMatrix();
		gradientMatrix.setTranslate(-50, 0);

		Matrix matrix = new Matrix();
		matrix.setScale(2, 1);

		VectorGraphicsPainter.fillGradientPath(g, path, matrix, gradient);
		display.flush();

		TestUtilities.check("left", 0, 0, g, Colors.RED);
		TestUtilities.check("center", 99, 0, g, Colors.YELLOW);
		TestUtilities.check("right", 199, 0, g, Colors.WHITE);
	}

	/**
	 * Tests a rotated horizontal gradient in a translated path.
	 */
	@Test
	public static void testGradientHorizontalRotateGlobalTranslate() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		Path path = new Path();
		path.moveTo(0, 0);
		path.lineTo(100, 0);
		path.lineTo(100, 100);
		path.lineTo(0, 100);
		path.close();

		LinearGradient gradient = new LinearGradient(0, 0, 140, 0, new int[] { 0xffff0000, 0xffffff00, 0xffffffff },
				new float[] { 0.0f, 0.5f, 1.0f });
		Matrix gradientMatrix = gradient.getMatrix();
		gradientMatrix.setRotate(45);

		Matrix matrix = new Matrix();
		matrix.setTranslate(50, 50);

		VectorGraphicsPainter.fillGradientPath(g, path, matrix, gradient);
		display.flush();

		TestUtilities.check("left", 50, 50, g, Colors.RED);
		TestUtilities.check("right", 149, 149, g, Colors.WHITE);
	}

	/**
	 * Tests a scaled horizontal gradient in a translated and rotated path.
	 */
	@Test
	public static void testGradientHorizontalScaleGlobalRotate() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		Path path = new Path();
		path.moveTo(0, 0);
		path.lineTo(100, 0);
		path.lineTo(100, 100);
		path.lineTo(0, 100);
		path.close();

		LinearGradient gradient = new LinearGradient(0, 0, 49, 0, new int[] { 0xffff0000, 0xffffff00, 0xffffffff },
				new float[] { 0.0f, 0.5f, 1.0f });
		Matrix gradientMatrix = gradient.getMatrix();
		gradientMatrix.setScale(2, 2);

		Matrix matrix = new Matrix();
		matrix.setTranslate(50, 50);
		matrix.preRotate(45);

		VectorGraphicsPainter.fillGradientPath(g, path, matrix, gradient);
		display.flush();

		TestUtilities.check("left", 50, 51, g, Colors.RED);
		TestUtilities.check("right", 50, 189, g, Colors.WHITE);
	}

	/**
	 * Tests an horizontal gradient with a fully transparent alpha.
	 */
	@Test
	public static void testGradientColorTransparent() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		Path path = new Path();
		path.moveTo(0, 0);
		path.lineTo(100, 0);
		path.lineTo(100, 100);
		path.lineTo(0, 100);
		path.close();

		LinearGradient gradient = new LinearGradient(0, 0, 99, 0, new int[] { 0xffff0000, 0xffffff00, 0xffffffff });

		VectorGraphicsPainter.fillGradientPath(g, path, new Matrix(), gradient, FillType.WINDING, 0x00,
				BlendMode.SRC_OVER);
		display.flush();

		TestUtilities.check("left", 0, 0, g, BACKGROUND_COLOR);
		TestUtilities.check("center", 49, 0, g, BACKGROUND_COLOR);
		TestUtilities.check("right", 99, 0, g, BACKGROUND_COLOR);
	}

	/**
	 * Tests an horizontal gradient drawn with a partially transparent alpha.
	 */
	@Test
	public static void testGradientColorSemiTransparent() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();
		TestUtilities.clearScreen(Colors.BLACK);

		Path path = new Path();
		path.moveTo(0, 0);
		path.lineTo(100, 0);
		path.lineTo(100, 100);
		path.lineTo(0, 100);
		path.close();

		LinearGradient gradient = new LinearGradient(0, 0, 99, 0, new int[] { 0xffff0000, 0xffffff00, 0xffffffff });

		VectorGraphicsPainter.fillGradientPath(g, path, new Matrix(), gradient, FillType.WINDING, 0x88,
				BlendMode.SRC_OVER);
		display.flush();

		TestUtilities.check("left", 0, 0, g, 0x800000);
		TestUtilities.check("center", 49, 0, g, 0x808000);
		TestUtilities.check("right", 99, 0, g, 0x808080);
	}

	/**
	 * Tests a partially transparent horizontal gradient drawn with a partially transparent alpha.
	 */
	@Test
	public static void testGradientSemiTransparentColorSemiTransparent() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();
		TestUtilities.clearScreen(Colors.BLACK);

		Path path = new Path();
		path.moveTo(0, 0);
		path.lineTo(100, 0);
		path.lineTo(100, 100);
		path.lineTo(0, 100);
		path.close();

		LinearGradient gradient = new LinearGradient(0, 0, 99, 0, new int[] { 0x88ff0000, 0x88ffff00, 0x88ffffff });

		VectorGraphicsPainter.fillGradientPath(g, path, new Matrix(), gradient, FillType.WINDING, 0x88,
				BlendMode.SRC_OVER);
		display.flush();

		TestUtilities.check("left", 0, 0, g, 0x490000);
		TestUtilities.check("center", 49, 0, g, 0x494900);
		TestUtilities.check("right", 99, 0, g, 0x494949);
	}

	/**
	 * Tests a gradient with the two points at the same place.
	 */
	@Test
	public static void testGradientTwoPointsAtSamePlace() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		Path path = new Path();
		path.moveTo(0, 0);
		path.lineTo(100, 0);
		path.lineTo(100, 100);
		path.lineTo(0, 100);
		path.close();

		LinearGradient gradient = new LinearGradient(49, 49, 49, 49, new int[] { 0xffff0000, 0xff880088, 0xff0000ff });

		VectorGraphicsPainter.fillGradientPath(g, path, new Matrix(), gradient);
		display.flush();

		TestUtilities.check("left", 0, 0, g, Colors.RED);
		TestUtilities.check("centerleft", 48, 0, g, Colors.RED);
		TestUtilities.check("centerright", 50, 0, g, Colors.BLUE);
		TestUtilities.check("right", 99, 0, g, Colors.BLUE);
	}

	/**
	 * Tests a gradient path with clipping: a clip larger than the path.
	 */
	@Test
	public static void testGradientWithLargeClip() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		int x = 100;
		int y = 100;
		int width = 60;
		int height = 60;
		int testThickness = 1;

		int errors = 0;

		Path path = new Path();
		path.moveTo(0, 0);
		path.lineToRelative(0, height);
		path.lineToRelative(width, 0);
		path.lineToRelative(0, -height);
		path.close();

		// Gradient with same color at start and end
		LinearGradient gradient = new LinearGradient(0, 0, width, 0, new int[] { 0xff0000ff, 0xff0000ff });

		Matrix m = new Matrix();
		m.setTranslate(x, y); // Brings top left to x,y

		// Clip larger than path
		int outerClip = 10;
		g.setClip(x - outerClip, y - outerClip, width + 2 * outerClip, height + 2 * outerClip);

		g.setColor(Colors.BLUE);
		VectorGraphicsPainter.fillGradientPath(g, path, m, gradient);
		display.flush();

		errors += TestUtilities.checkPeripheralArea("clip over inside", Colors.BLUE, x + testThickness,
				y + testThickness, width - 2 * testThickness, height - 2 * testThickness, testThickness, 0, false);
		errors += TestUtilities.checkPeripheralArea("clip over outside", BACKGROUND_COLOR, x, y, width, height,
				testThickness, 0, false);
		assertEquals(0, errors);
	}

	/**
	 * Tests a gradient path with clipping: a clip that just fits the path.
	 */
	@Test
	public static void testGradientWithFitClip() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		int x = 100;
		int y = 100;
		int width = 60;
		int height = 60;
		int testThickness = 1;

		int errors = 0;

		Path path = new Path();
		path.moveTo(0, 0);
		path.lineToRelative(0, height);
		path.lineToRelative(width, 0);
		path.lineToRelative(0, -height);
		path.close();

		// Gradient with same color at start and end
		LinearGradient gradient = new LinearGradient(0, 0, width, 0, new int[] { 0xff0000ff, 0xff0000ff });

		Matrix m = new Matrix();
		m.setTranslate(x, y); // Brings top left to x,y

		int outerClip = 0;
		g.setClip(x - outerClip, y - outerClip, width + 2 * outerClip, height + 2 * outerClip);

		g.setColor(Colors.BLUE);
		VectorGraphicsPainter.fillGradientPath(g, path, m, gradient);
		display.flush();

		errors += TestUtilities.checkPeripheralArea("clip fit inside", Colors.BLUE, x + testThickness,
				y + testThickness, width - 2 * testThickness, height - 2 * testThickness, testThickness, 0, false);
		errors += TestUtilities.checkPeripheralArea("clip fit outside", BACKGROUND_COLOR, x, y, width, height,
				testThickness, 0, false);
		assertEquals(0, errors);
	}

	/**
	 * Tests a gradient path with clipping: a clip smaller than the path.
	 */
	@Test
	public static void testGradientWithThinClip() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		int x = 100;
		int y = 100;
		int width = 60;
		int height = 60;
		int testThickness = 1;

		int errors = 0;

		Path path = new Path();
		path.moveTo(0, 0);
		path.lineToRelative(0, height);
		path.lineToRelative(width, 0);
		path.lineToRelative(0, -height);
		path.close();

		// Gradient with same color at start and end
		LinearGradient gradient = new LinearGradient(0, 0, width, 0, new int[] { 0xff0000ff, 0xff0000ff });

		Matrix m = new Matrix();
		m.setTranslate(x, y); // Brings top left to x,y

		int outerClip = -10;
		g.setClip(x - outerClip, y - outerClip, width + 2 * outerClip, height + 2 * outerClip);

		g.setColor(Colors.BLUE);
		VectorGraphicsPainter.fillGradientPath(g, path, m, gradient);
		display.flush();

		errors += TestUtilities.checkPeripheralArea("clip inner inside", Colors.BLUE, x - outerClip + testThickness,
				y - outerClip + testThickness, width + 2 * outerClip - 2 * testThickness,
				height + 2 * outerClip - 2 * testThickness, testThickness, 0, false);
		errors += TestUtilities.checkPeripheralArea("clip inner outside", BACKGROUND_COLOR, x - outerClip,
				y - outerClip, width + 2 * outerClip, height + 2 * outerClip, testThickness, 0, false);
		assertEquals(0, errors);
	}

	/**
	 * Tests a translated square path (gc and matrix).
	 */
	@Test
	public static void testGCTranslateMatrixTranslate() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		Path path = new Path();
		path.moveTo(0, 0);
		path.lineTo(100, 0);
		path.lineTo(100, 100);
		path.lineTo(0, 100);
		path.close();

		LinearGradient gradient = new LinearGradient(0, 0, 99, 0, new int[] { 0xffff0000, 0xffffff00, 0xffffffff });

		Matrix matrix = new Matrix();
		matrix.setTranslate(0, 100);
		g.translate(100, 0);

		g.setColor(Colors.RED);
		VectorGraphicsPainter.fillGradientPath(g, path, matrix, gradient);
		display.flush();
		g.resetTranslation();

		TestUtilities.check("left", 100, 100, g, Colors.RED);
		TestUtilities.check("center", 149, 149, g, Colors.YELLOW);
		TestUtilities.check("right", 199, 199, g, Colors.WHITE);
	}

	/**
	 * Tests a translated (gc) and scaled (matrix) square path.
	 */
	@Test
	public static void testGCTranslateMatrixScale() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		int size = 60;
		Path path = new Path();
		path.moveTo(0, 0);
		path.lineTo(size, 0);
		path.lineTo(size, size);
		path.lineTo(0, size);
		path.close();

		LinearGradient gradient = new LinearGradient(0, 0, size - 1, 0,
				new int[] { 0xffff0000, 0xffffff00, 0xffffffff });

		int sx = 3;
		int sy = 2;
		Matrix matrix = new Matrix();
		matrix.setScale(sx, sy);

		int tx = 10;
		int ty = 10;
		g.translate(tx, ty);

		g.setColor(Colors.RED);
		VectorGraphicsPainter.fillGradientPath(g, path, matrix, gradient);
		display.flush();
		g.resetTranslation();

		TestUtilities.check("top left", tx, ty, g, Colors.RED);
		TestUtilities.check("center", tx + (size * sx) / 2 - 1, ty + (size * sy) / 2 - 1, g, Colors.YELLOW);
		TestUtilities.check("bottom right", tx + (size * sx) - 1, ty + (size * sy) - 1, g, Colors.WHITE);
	}

	/**
	 * Tests a translated (gc) and rotated (matrix) square path.
	 */
	@Test
	public static void testGCTranslateMatrixRotate() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		Path path = new Path();
		path.moveTo(0, 0);
		path.lineTo(100, 0);
		path.lineTo(100, 100);
		path.lineTo(0, 100);
		path.close();

		LinearGradient gradient = new LinearGradient(0, 0, 99, 0, new int[] { 0xffff0000, 0xffffff00, 0xffffffff });

		Matrix matrix = new Matrix();
		matrix.setRotate(45);

		g.translate(100, 100);

		g.setColor(Colors.RED);
		VectorGraphicsPainter.fillGradientPath(g, path, matrix, gradient);
		display.flush();
		g.resetTranslation();

		TestUtilities.check("top", 100, 101, g, Colors.RED);
		TestUtilities.check("left", 30, 170, g, Colors.RED);
		TestUtilities.check("right", 169, 170, g, Colors.WHITE);
		TestUtilities.check("bottom", 100, 239, g, Colors.WHITE);
	}

	/**
	 * An AVD image with a gradient without item and color: expects transparent.
	 */
	@Test
	public static void testAVDGradientNoItemNoColor() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		VectorImage image = VectorImage.getImage("/com/microej/microvg/test/TestGradient_gradient_0_item_0_color.xml");
		VectorGraphicsPainter.drawImage(g, image, new Matrix());
		display.flush();

		TestUtilities.check("left", 0, 0, g, BACKGROUND_COLOR);
		TestUtilities.check("center", 49, 0, g, BACKGROUND_COLOR);
		TestUtilities.check("right", 99, 0, g, BACKGROUND_COLOR);
	}

	/**
	 * An AVD image with a gradient without item and with one color: expects one color.
	 */
	@Test
	public static void testAVDGradientNoItemOneColor() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		VectorImage image = VectorImage.getImage("/com/microej/microvg/test/TestGradient_gradient_0_item_1_color.xml");
		VectorGraphicsPainter.drawImage(g, image, new Matrix());
		display.flush();

		TestUtilities.check("left", 0, 0, g, 0xff00ff);
		TestUtilities.check("center", 49, 0, g, 0xff00ff);
		TestUtilities.check("right", 99, 0, g, 0xff00ff);
	}

	/**
	 * An AVD image with a gradient without item and with two colors: expects two colors (vertical).
	 */
	@Test
	public static void testAVDGradientNoItemTwoColors() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		VectorImage image = VectorImage.getImage("/com/microej/microvg/test/TestGradient_gradient_0_item_2_colors.xml");
		VectorGraphicsPainter.drawImage(g, image, new Matrix());
		display.flush();

		TestUtilities.check("top", 0, 0, g, 0x0000ff);
		TestUtilities.check("middle", 0, 49, g, 0x8000ff);
		TestUtilities.check("bottom", 0, 99, g, 0xff00ff);
	}

	/**
	 * An AVD image with a gradient without item and with three colors: expects three colors (vertical).
	 */
	@Test
	public static void testAVDGradientNoItemThreeColors() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		VectorImage image = VectorImage.getImage("/com/microej/microvg/test/TestGradient_gradient_0_item_3_colors.xml");
		VectorGraphicsPainter.drawImage(g, image, new Matrix());
		display.flush();

		TestUtilities.check("top", 0, 0, g, 0x0000ff);
		TestUtilities.check("middle", 0, 49, g, 0x00ff00);
		TestUtilities.check("bottom", 0, 99, g, 0xff00ff);
	}

	/**
	 * An AVD image with a gradient with one item and without color: expects one color.
	 */
	@Test
	public static void testAVDGradientOneItemNoColor() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		VectorImage image = VectorImage.getImage("/com/microej/microvg/test/TestGradient_gradient_1_item_0_color.xml");
		VectorGraphicsPainter.drawImage(g, image, new Matrix());
		display.flush();

		TestUtilities.check("left", 0, 0, g, 0xff0000);
		TestUtilities.check("right", 49, 0, g, 0xff0000);
		TestUtilities.check("center", 99, 0, g, 0xff0000);
	}

	/**
	 * An AVD image with a gradient with two items and without color: expects two colors (horizontal).
	 */
	@Test
	public static void testAVDGradientTwoItemsNoColor() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		VectorImage image = VectorImage.getImage("/com/microej/microvg/test/TestGradient_gradient_2_items_0_color.xml");
		VectorGraphicsPainter.drawImage(g, image, new Matrix());
		display.flush();

		TestUtilities.check("left", 0, 0, g, 0xff0000);
		TestUtilities.check("center", 49, 0, g, 0x808000);
		TestUtilities.check("right", 99, 0, g, 0x00ff00);
	}

	/**
	 * A SVG image with a gradient without stop: expects transparent.
	 */
	@Test
	public static void testSVGGradientNoStop() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		VectorImage image = VectorImage.getImage("/com/microej/microvg/test/TestGradient_0_stop.svg");
		VectorGraphicsPainter.drawImage(g, image, new Matrix());
		display.flush();

		TestUtilities.check("left", 0, 0, g, BACKGROUND_COLOR);
		TestUtilities.check("center", 49, 0, g, BACKGROUND_COLOR);
		TestUtilities.check("right", 99, 0, g, BACKGROUND_COLOR);
	}

	/**
	 * A SVG image with a gradient with one stop: expects one color.
	 */
	@Test
	public static void testSVGGradientOneStop() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		VectorImage image = VectorImage.getImage("/com/microej/microvg/test/TestGradient_1_stop.svg");
		VectorGraphicsPainter.drawImage(g, image, new Matrix());
		display.flush();

		TestUtilities.check("left", 0, 0, g, 0xff00ff);
		TestUtilities.check("center", 49, 0, g, 0xff00ff);
		TestUtilities.check("right", 99, 0, g, 0xff00ff);
	}

	/**
	 * A SVG image with a gradient with two stops: expects two colors (diagonal).
	 */
	@Test
	public static void testSVGGradientTwoStops() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		VectorImage image = VectorImage.getImage("/com/microej/microvg/test/TestGradient_2_stops.svg");
		VectorGraphicsPainter.drawImage(g, image, new Matrix());
		display.flush();

		TestUtilities.check("left", 0, 0, g, 0x0000ff);
		TestUtilities.check("center", 49, 49, g, 0x8000ff);
		TestUtilities.check("right", 99, 99, g, 0xff00ff);
	}

	/**
	 * A SVG image with a gradient with three stops: expects three colors (diagonal).
	 */
	@Test
	public static void testSVGGradientThreeStops() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		VectorImage image = VectorImage.getImage("/com/microej/microvg/test/TestGradient_3_stops.svg");
		VectorGraphicsPainter.drawImage(g, image, new Matrix());
		display.flush();

		TestUtilities.check("left", 0, 0, g, 0x0000ff);
		TestUtilities.check("center", 49, 49, g, 0x00ff00);
		TestUtilities.check("right", 99, 99, g, 0xff00ff);
	}

	/**
	 * A linear gradient without color: expects transparent.
	 */
	@Test
	public static void testLinearGradientNoColor() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		int[] colors = new int[0];
		float[] positions = new float[0];
		LinearGradient gradient = new LinearGradient(0, 0, 99, 99, colors, positions);
		VectorGraphicsPainter.fillGradientPath(g, getPath(), new Matrix(), gradient);
		display.flush();

		TestUtilities.check("left", 0, 0, g, BACKGROUND_COLOR);
		TestUtilities.check("center", 49, 49, g, BACKGROUND_COLOR);
		TestUtilities.check("right", 99, 99, g, BACKGROUND_COLOR);
	}

	/**
	 * A linear gradient with one color: expects one color.
	 */
	@Test
	public static void testLinearGradientOneColor() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		int[] colors = new int[] { 0xffff0000 };
		float[] positions = new float[] { 1 };
		LinearGradient gradient = new LinearGradient(0, 0, 99, 99, colors, positions);
		VectorGraphicsPainter.fillGradientPath(g, getPath(), new Matrix(), gradient);
		display.flush();

		TestUtilities.check("left", 0, 0, g, Colors.RED);
		TestUtilities.check("center", 49, 49, g, Colors.RED);
		TestUtilities.check("right", 99, 99, g, Colors.RED);
	}

	/**
	 * A linear gradient with two colors: expects two colors.
	 */
	@Test
	public static void testLinearGradientTwoColors() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		int[] colors = new int[] { 0xffff0000, 0xff0000ff };
		float[] positions = new float[] { 0, 1 };
		LinearGradient gradient = new LinearGradient(0, 0, 99, 99, colors, positions);
		VectorGraphicsPainter.fillGradientPath(g, getPath(), new Matrix(), gradient);
		display.flush();

		TestUtilities.check("left", 0, 0, g, Colors.RED);
		TestUtilities.check("center", 49, 49, g, 0x800080);
		TestUtilities.check("right", 99, 99, g, Colors.BLUE);
	}

	/**
	 * A linear gradient without color: expects transparent.
	 */
	@Test
	public static void testBVILinearGradientNoColor() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();
		try (BufferedVectorImage bvi = new BufferedVectorImage(150, 150)) {
			GraphicsContext bgc = bvi.getGraphicsContext();

			int[] colors = new int[0];
			float[] positions = new float[0];
			LinearGradient gradient = new LinearGradient(0, 0, 99, 99, colors, positions);
			VectorGraphicsPainter.fillGradientPath(bgc, getPath(), new Matrix(), gradient);
			VectorGraphicsPainter.drawImage(g, bvi, new Matrix());
			display.flush();

			TestUtilities.check("left", 0, 0, g, BACKGROUND_COLOR);
			TestUtilities.check("center", 49, 49, g, BACKGROUND_COLOR);
			TestUtilities.check("right", 99, 99, g, BACKGROUND_COLOR);
		}
	}

	/**
	 * A linear gradient with one color: expects one color.
	 */
	@Test
	public static void testBVILinearGradientOneColor() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();
		try (BufferedVectorImage bvi = new BufferedVectorImage(150, 150)) {
			GraphicsContext bgc = bvi.getGraphicsContext();

			int[] colors = new int[] { 0xffff0000 };
			float[] positions = new float[] { 1 };
			LinearGradient gradient = new LinearGradient(0, 0, 99, 99, colors, positions);
			VectorGraphicsPainter.fillGradientPath(bgc, getPath(), new Matrix(), gradient);
			VectorGraphicsPainter.drawImage(g, bvi, new Matrix());
			display.flush();

			TestUtilities.check("left", 0, 0, g, Colors.RED);
			TestUtilities.check("center", 49, 49, g, Colors.RED);
			TestUtilities.check("right", 99, 99, g, Colors.RED);
		}
	}

	/**
	 * A linear gradient with two colors: expects two colors.
	 */
	@Test
	public static void testBVILinearGradientTwoColors() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();
		try (BufferedVectorImage bvi = new BufferedVectorImage(150, 150)) {
			GraphicsContext bgc = bvi.getGraphicsContext();

			int[] colors = new int[] { 0xffff0000, 0xff0000ff };
			float[] positions = new float[] { 0, 1 };
			LinearGradient gradient = new LinearGradient(0, 0, 99, 99, colors, positions);
			VectorGraphicsPainter.fillGradientPath(bgc, getPath(), new Matrix(), gradient);
			VectorGraphicsPainter.drawImage(g, bvi, new Matrix());
			display.flush();

			TestUtilities.check("left", 0, 0, g, Colors.RED);
			TestUtilities.check("center", 49, 49, g, 0x800080);
			TestUtilities.check("right", 99, 99, g, Colors.BLUE);
		}
	}

	private static Path getPath() {
		Path path = new Path();
		path.moveTo(0, 0);
		path.lineTo(100, 0);
		path.lineTo(100, 100);
		path.lineTo(0, 100);
		path.close();
		return path;
	}
}
