/*
 * Java
 *
 * Copyright 2024-2025 MicroEJ Corp. All rights reserved.
 * MicroEJ Corp. PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package ej.microvg.paint;

import java.awt.Color;
import java.awt.LinearGradientPaint;
import java.awt.Paint;
import java.awt.geom.AffineTransform;
import java.awt.geom.Point2D;

import ej.microvg.MatrixHelper;

/**
 * Represents a path painter with a linear gradient on a vector.
 */
public class LinearGradient implements VGPaint {

	/**
	 * The gradient's colors.
	 */
	private final Color[] colors;

	/**
	 * The positions of the gradient's colors on the vector.
	 */
	private final float[] fractions;

	/**
	 * The deformation of the gradient to apply at drawing time.
	 */
	private final AffineTransform at;

	/**
	 * Creates a painter over the given linear gradient.
	 *
	 * @param colors
	 *            array of colors
	 * @param fractions
	 *            array of fractions
	 * @param matrix
	 *            gradient vector transformation
	 * @throws IllegalArgumentException
	 *             whether the number of colors is different than the number of fractions.
	 */
	public LinearGradient(Color[] colors, float[] fractions, float[] matrix) {
		this(colors, fractions, MatrixHelper.getAffineTransform(matrix, 0, 0));
	}

	/**
	 * Creates a painter over the given linear gradient.
	 *
	 * @param colors
	 *            array of colors
	 * @param fractions
	 *            array of fractions
	 * @param xStart
	 *            the x-coordinate for the start of the gradient vector
	 * @param yStart
	 *            the y-coordinate for the start of the gradient vector
	 * @param xEnd
	 *            the x-coordinate for the end of the gradient vector
	 * @param yEnd
	 *            the y-coordinate for the end of the gradient vector
	 * @throws IllegalArgumentException
	 *             whether the number of colors is different than the number of fractions.
	 */
	public LinearGradient(Color[] colors, float[] fractions, float xStart, float yStart, float xEnd, float yEnd) {
		this(colors, fractions, transformMatrix(MatrixHelper.createIdentity(), xStart, yStart, xEnd, yEnd));
	}

	/**
	 * Creates a painter over the given linear gradient.
	 *
	 * @param colors
	 *            array of colors
	 * @param fractions
	 *            array of fractions
	 * @param matrix
	 *            gradient vector transformation
	 * @param xStart
	 *            the x-coordinate for the start of the gradient vector
	 * @param yStart
	 *            the y-coordinate for the start of the gradient vector
	 * @param xEnd
	 *            the x-coordinate for the end of the gradient vector
	 * @param yEnd
	 *            the y-coordinate for the end of the gradient vector
	 * @throws IllegalArgumentException
	 *             whether the number of colors is different than the number of fractions.
	 */
	public LinearGradient(Color[] colors, float[] fractions, float[] matrix, float xStart, float yStart, float xEnd,
			float yEnd) {
		this(colors, fractions, transformMatrix(matrix.clone(), xStart, yStart, xEnd, yEnd));
	}

	/**
	 * Creates a painter over the given linear gradient.
	 *
	 * @param colors
	 *            array of colors
	 * @param fractions
	 *            array of fractions
	 * @param matrix
	 *            gradient vector transformation
	 * @throws IllegalArgumentException
	 *             whether the number of colors is different than the number of fractions.
	 */
	public LinearGradient(Color[] colors, float[] fractions, AffineTransform matrix) {
		if (colors.length != fractions.length) {
			throw new IllegalArgumentException();
		}
		this.colors = colors;
		this.fractions = fractions;
		this.at = matrix;
	}

	@Override
	public LinearGradient apply(PaintVisitor transformer) {

		// visit colors
		int length = this.colors.length;
		Color[] oldColors = this.colors;
		Color[] newColors = new Color[length];
		for (int i = 0; i < length; i++) {
			newColors[i] = new Color(transformer.visitColor(oldColors[i].getRGB()), true);
		}

		// visit transformation
		AffineTransform at = transformer.visitTransform(this.at);

		return new LinearGradient(newColors, this.fractions, at);
	}

	@Override
	public Paint getPaint() {
		final Point2D gradientOrigin = new Point2D.Float();
		final Point2D gradientEnd = new Point2D.Float();
		this.at.transform(new Point2D.Float(0f, 0f), gradientOrigin);
		this.at.transform(new Point2D.Float(1f, 0f), gradientEnd);
		return new LinearGradientPaint(gradientOrigin, gradientEnd, this.fractions, this.colors);
	}

	private static float[] transformMatrix(float[] matrix, float x0, float y0, float x1, float y1) {
		MatrixHelper.translate(matrix, x0, y0);
		MatrixHelper.rotate(matrix, getGradientAngle(x0, y0, x1, y1));
		MatrixHelper.scale(matrix,
				Math.max(getGradientLength(x0, y0, x1, y1), 0.1f /* very small length to avoid scaling to 0 */),
				1 /* no scale on Y */);
		return matrix;
	}

	private static float getGradientLength(float x0, float y0, float x1, float y1) {
		return (float) Math.sqrt(Math.pow(x1 - x0, 2) + Math.pow(y1 - y0, 2));
	}

	private static float getGradientAngle(float x0, float y0, float x1, float y1) {
		double angle = Math.atan2(y1 - y0, x1 - x0);
		return (float) Math.toDegrees(angle);
	}

}
