/*
 * Java
 *
 * Copyright 2025 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package com.microej.kf.util.control.net;

import com.microej.kf.util.module.SandboxedModule;
import com.microej.kf.util.module.SandboxedModuleHelper;
import ej.service.ServiceFactory;

import java.net.Socket;

/**
 * Action performed when reading on a socket.
 */
public class ReadSocket extends AbstractSocket {

	private final int nbBytes;

	/**
	 * Creates a read socket action.
	 *
	 * @param socket
	 * 		the socket of the read action.
	 * @param nbBytes
	 * 		the number of read bytes.
	 */
	public ReadSocket(Socket socket, int nbBytes) {
		super(socket);
		this.nbBytes = nbBytes;
	}

	/**
	 * Gets the number of bytes to be read.
	 *
	 * @return the number of bytes to be read.
	 */
	public int getNbBytes() {
		return this.nbBytes;
	}

	/**
	 * Called when the execution of the action is about to start.
	 *
	 * This hook is used to check if the action is allowed to be performed.
	 *
	 * @return the number of bytes read.
	 * @throws SecurityException
	 * 		if the action is not allowed to be performed.
	 */
	public int onStart() throws SecurityException {
		SandboxedModuleHelper cmh = ServiceFactory.getRequiredService(SandboxedModuleHelper.class);
		SandboxedModule currentModule = cmh.getCurrentModule();
		return currentModule.getNetworkResourceController().onStart(currentModule, this);
	}

	/**
	 * Called when the execution of the action is finished.
	 *
	 * This hook is used to update the current resources state of the module that performs the action.
	 *
	 * @param nbBytesRead
	 * 		the actual number of bytes read.
	 */
	public void onEnd(int nbBytesRead) {
		SandboxedModuleHelper cmh = ServiceFactory.getRequiredService(SandboxedModuleHelper.class);
		SandboxedModule currentModule = cmh.getCurrentModule();
		currentModule.getNetworkResourceController().onEnd(currentModule, this, nbBytesRead);
	}

}
