/*
 * Java
 *
 * Copyright 2012-2025 MicroEJ Corp. All rights reserved.
 * MicroEJ Corp. PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package ej.microui.event;

import ej.fp.FrontPanel;

/**
 * This class holds all methods to send MicroUI generic events to MicroUI application.
 */
@SuppressWarnings("nls")
public class EventGeneric {

	private EventGeneric() {
		// hide implicit constructor
	}

	/**
	 * Send a generic event to the event generator.
	 * <p>
	 * The result of this method is unknown when the given event generator is not a MicroUI <code>EventGenerator</code>
	 * event generator.
	 *
	 * @param eventGenerator
	 *            the MicroUI <code>EventGenerator</code> event generator.
	 * @param event
	 *            the 32-bits event.
	 */
	public static void sendEvent(String eventGenerator, int event) {
		int[] events = new int[2];
		events[1] = event;
		sendUserEvents(eventGenerator, events);
	}

	/**
	 * Send several generic events to the event generator.
	 * <p>
	 * The result of this method is unknown when the given event generator is not a MicroUI <code>EventGenerator</code>
	 * event generator.
	 * <p>
	 * Only the first 255 events are sent.
	 *
	 * @param eventGenerator
	 *            the MicroUI <code>EventGenerator</code> event generator.
	 * @param eventsArray
	 *            the address of the 32-bits events array.
	 */
	public static void sendEvents(String eventGenerator, int[] eventsArray) {
		int length = Math.min(eventsArray.length, 256); // spec
		int[] events = new int[length + 1];
		System.arraycopy(eventsArray, 0, events, 1, length);
		sendUserEvents(eventGenerator, events);
	}

	private static void sendUserEvents(String eventGenerator, int[] events) {
		int eventsLength = events.length;
		try {
			events[0] = LLUIInput.buildEvent(eventGenerator, eventsLength - 1);
			LLUIInput.sendEvents(events);
		} catch (IllegalArgumentException e) {
			// invalid MicroUI events generator ID
			// -> print a trace
			StringBuilder buf = new StringBuilder();
			buf.append("[");
			buf.append(EventGeneric.class.getName());
			buf.append("] send events : {");
			for (int i = 0; ++i < eventsLength;) {
				buf.append("0x");
				buf.append(Integer.toHexString(events[i]));
				if (i != eventsLength - 1) {
					buf.append(", ");
				}
			}
			buf.append("} (");
			buf.append(eventGenerator);
			buf.append(")");
			FrontPanel.out.println(buf.toString());
		}
	}
}
