/*
 * Java
 *
 * Copyright 2024-2025 MicroEJ Corp. All rights reserved.
 * MicroEJ Corp. PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package ej.microui.display;

/**
 * This interface provides all the drawing methods that use a font.
 * <p>
 * This interface can be fully implemented by a dedicated class in front panel project. This allows to draw and decode
 * fonts with a custom format.
 */
public interface UIFontDrawing {

	/**
	 * Gets the supported custom font format.
	 *
	 * @return the supported custom font format
	 */
	MicroUIFontFormat handledFormat();

	/**
	 * Computes the rendered width of a string.
	 *
	 * @param string
	 *            a string of characters
	 * @param font
	 *            the MicroUI Font to use
	 * @return the width in pixels
	 */
	default int stringWidth(char[] string, MicroUIFont font) {
		return 0;
	}

	/**
	 * Draws a string.
	 *
	 * @param gc
	 *            the targeted MicroUI GraphicsContext
	 * @param string
	 *            a string of characters
	 * @param font
	 *            the MicroUI Font to use
	 * @param x
	 *            the left coordinate
	 * @param y
	 *            the top coordinate
	 */
	default void draw(MicroUIGraphicsContext gc, char[] string, MicroUIFont font, int x, int y) {
		gc.reportError(LLUIPainter.DRAWING_LOG_NOT_IMPLEMENTED);
	}

	/**
	 * Draws a character.
	 *
	 * @param gc
	 *            the targeted MicroUI GraphicsContext
	 * @param c
	 *            a character
	 * @param font
	 *            the MicroUI Font to use
	 * @param x
	 *            the left coordinate
	 * @param y
	 *            the top coordinate
	 */
	default void draw(MicroUIGraphicsContext gc, char c, MicroUIFont font, int x, int y) {
		gc.reportError(LLUIPainter.DRAWING_LOG_NOT_IMPLEMENTED);
	}

	/**
	 * Computes the rendered width of a string and fills the SNI context.
	 *
	 * @param string
	 *            a string of characters
	 * @param font
	 *            the MicroUI Font to use
	 * @param renderableString
	 *            the renderable string
	 * @return the width in pixels
	 */
	default int initializeRenderableStringSNIContext(char[] string, MicroUIFont font,
			MicroUIRenderableString renderableString) {
		return stringWidth(string, font);
	}

	/**
	 * Draws a renderable string.
	 *
	 * @param gc
	 *            the targeted MicroUI GraphicsContext
	 * @param string
	 *            a string of characters
	 * @param font
	 *            the MicroUI Font to use
	 * @param charsWidth
	 *            the string width
	 * @param renderableString
	 *            the renderable string
	 * @param x
	 *            the left coordinate
	 * @param y
	 *            the top coordinate
	 */
	default void drawRenderableString(MicroUIGraphicsContext gc, char[] string, MicroUIFont font, int charsWidth,
			MicroUIRenderableString renderableString, int x, int y) {
		draw(gc, string, font, x, y);
	}

	/**
	 * Draws a string applying a rotation and an alpha value.
	 *
	 * This method uses the bilinear algorithm to render the character. This algorithm performs better rendering than
	 * nearest neighbor algorithm but it is slower to apply.
	 *
	 * @param gc
	 *            the targeted MicroUI GraphicsContext
	 * @param c
	 *            a character
	 * @param font
	 *            the MicroUI Font to use
	 * @param destX
	 *            the left coordinate
	 * @param destY
	 *            the top coordinate
	 * @param xRotation
	 *            the horizontal coordinate of the rotation center
	 * @param yRotation
	 *            the vertical coordinate of the rotation center
	 * @param angle
	 *            the rotation angle, in degrees
	 * @param alpha
	 *            the alpha value to use
	 * @param fontData
	 *            the MicroUI font to use
	 */
	default void drawCharWithRotationBilinear(MicroUIGraphicsContext gc, char c, // NOSONAR avoid to create object here
			MicroUIFont font, int destX, int destY, int xRotation, int yRotation, float angle, int alpha) {
		gc.reportError(LLUIPainter.DRAWING_LOG_NOT_IMPLEMENTED);
	}

	/**
	 * Draws a string applying a rotation and an alpha value.
	 *
	 * This method uses the nearest neighbor algorithm to render the character. This algorithm performs faster rendering
	 * than nearest neighbor algorithm but gives a rougher result.
	 *
	 * @param gc
	 *            the targeted MicroUI GraphicsContext
	 * @param c
	 *            a character
	 * @param font
	 *            the MicroUI Font to use
	 * @param destX
	 *            the left coordinate
	 * @param destY
	 *            the top coordinate
	 * @param xRotation
	 *            the horizontal coordinate of the rotation center
	 * @param yRotation
	 *            the vertical coordinate of the rotation center
	 * @param angle
	 *            the rotation angle, in degrees
	 * @param alpha
	 *            the alpha value to use
	 * @param fontData
	 *            the MicroUI font to use
	 */
	default void drawCharWithRotationNearestNeighbor(MicroUIGraphicsContext gc, // NOSONAR avoid to create object here
			char c, MicroUIFont font, int destX, int destY, int xRotation, int yRotation, float angle, int alpha) {
		gc.reportError(LLUIPainter.DRAWING_LOG_NOT_IMPLEMENTED);
	}

	/**
	 * Draws a string applying a scaling.
	 *
	 * This method uses the bilinear algorithm to render the characters. This algorithm performs better rendering than
	 * nearest neighbor algorithm but it is slower to apply.
	 *
	 * @param gc
	 *            the targeted MicroUI GraphicsContext
	 * @param string
	 *            a string of characters
	 * @param font
	 *            the MicroUI Font to use
	 * @param x
	 *            the left coordinate
	 * @param y
	 *            the top coordinate
	 * @param xRatio
	 *            the horizontal scaling ratio (1.0f meaning no scaling)
	 * @param yRatio
	 *            the vertical scaling ratio (1.0f meaning no scaling)
	 */
	default void drawScaledStringBilinear(MicroUIGraphicsContext gc, char[] string, MicroUIFont font, int x, int y,
			float xRatio, float yRatio) {
		gc.reportError(LLUIPainter.DRAWING_LOG_NOT_IMPLEMENTED);
	}

	/**
	 * Draws a renderable string applying a scaling.
	 *
	 * This method uses the bilinear algorithm to render the characters. This algorithm performs better rendering than
	 * nearest neighbor algorithm but it is slower to apply.
	 *
	 * @param gc
	 *            the targeted MicroUI GraphicsContext
	 * @param string
	 *            a string of characters
	 * @param font
	 *            the MicroUI Font to use
	 * @param charsWidth
	 *            the string width
	 * @param renderableString
	 *            the renderable string
	 * @param x
	 *            the left coordinate
	 * @param y
	 *            the top coordinate
	 * @param xRatio
	 *            the horizontal scaling ratio (1.0f meaning no scaling)
	 * @param yRatio
	 *            the vertical scaling ratio (1.0f meaning no scaling)
	 */
	default void drawScaledRenderableStringBilinear(MicroUIGraphicsContext gc, // NOSONAR avoid to create object here
			char[] string, MicroUIFont font, int charsWidth, MicroUIRenderableString renderableString, int x, int y,
			float xRatio, float yRatio) {
		drawScaledStringBilinear(gc, string, font, x, y, xRatio, yRatio);
	}

}
